/*
 * Many of the character bitmaps below formed the typeface embodied in
 * the SAA5050 series of character-generator chips originally made and
 * sold by British company Mullard in the early 1980s.  Copyright in the
 * typeface will still be owned by Mullard's corporate successors, but
 * under section 55 of the Copyright Designs and Patents Act 1988 that
 * copyright is no longer infringed by the production or use of
 * articles specifically designed or adapted for producing material in
 * that typeface.
 *
 * The rest of the glyphs, and all of the code in this file, were
 * written by Ben Harris <bjh21@bjh21.me.uk>, Simon Tatham
 * <anakin@pobox.com>, and Marnanel Thurman <marnanel@thurman.org.uk>
 * between 2009 and 2020.
 *
 * To the extent possible under law, Ben Harris, Simon Tatham, and
 * Marnanel Thurman have dedicated all copyright and related and
 * neighboring rights to this software and the embodied typeface to
 * the public domain worldwide.  This software and typeface are
 * distributed without any warranty.
 *
 * You should have received a copy of the CC0 Public Domain Dedication
 * along with this software. If not, see
 * <http://creativecommons.org/publicdomain/zero/1.0/>.
 */
/*
 * This is a program to construct an outline font from a bitmap.  It's
 * based on the character-rounding algorithm of the Mullard SAA5050
 * series of Teletext character generators, and thus works best on
 * character shapes in the same style of those of the SAA5050.  This
 * file includes all of the glyphs from the SAA5050, SAA5051, SAA5052,
 * SAA5053, SAA5054, SAA5055, SAA5056, and SAA5057.  The output is a
 * Spline Font Database file suitable for feeding to Fontforge.
 *
 * The character-smoothing algorithm of the SAA5050 and friends is
 * a fairly simple means of expanding a 5x9 pixel character to 10x18
 * pixels for use on an interlaced display.  All it does is to detect
 * 2x2 clumps of pixels containing a diagonal line and add a couple of
 * subpixels to it, like this:
 *
 * . #  -> . . # # -> . . # # or # . -> # # . . -> # # . .
 * # .     . . # #    . # # #    . #    # # . .    # # # .
 *         # # . .    # # # .           . . # #    . # # #
 *         # # . .    # # . .           . . # #    . . # #
 *
 * This is applied to every occurrence of these patterns, even when
 * they overlap, and the result is that thin diagonal lines are
 * smoothed out while other features mostly remain the same.
 *
 * One way of extending this towards continuity would be to repeatedly
 * double the resolution and add more pixels to diagonals each time,
 * but this ends up with the diagonals being much too heavy.  Instead,
 * in places where the SAA5050 would add pixels, this program adds a
 * largeish triangle to each unfilled pixel, and removes a small
 * triangle from each filled one, something like this:
 *
 * . #  -> . . # # -> . . / # or # . -> # # . . -> # \ . .
 * # .     . . # #    . / # /    . #    # # . .    \ # \ .
 *         # # . .    / # / .           . . # #    . \ # \
 *         # # . .    # / . .           . . # #    . . \ #
 * 
 * The position of the lines is such that the diagonal stroke is the
 * same width as a horizontal or vertical stroke (assuming square
 * pixels).  There are a few additional complications, in that the
 * trimming of filled pixels can leave odd gaps where a diagonal stem
 * joins another one, so the code detects this and doesn't trim in
 * these cases:
 *
 * . # # -> . . # # # # -> . . / # # # -> . . / # # #
 * # . .    . . # # # #    . / # / # #    . / # # # #
 *          # # . . . .    / # / . . .    / # / . . .
 *          # # . . . .    # / . . . .    # / . . . .
 *
 * That is the interesting part of the program, and is in the dochar()
 * function.  Most of the rest is just dull geometry to join all the
 * bits together into a sensible outline.  Much of the code is wildly
 * inefficient -- O(n^2) algorithms aren't much of a problem when you
 * have at most a few thousand points to deal with.
 *
 * A rather nice feature of the outlines produced by this program is
 * that when rasterised at precisely 10 or 20 pixels high, they
 * produce the input and output respectively of the character-rounding
 * process.  While there are obious additional smoothings that could
 * be applied, doing so would probably lose this nice property.
 *
 * The glyph bitmaps included below include all the ones from the various
 * members of the SAA5050 family that I know about.  They are as shown
 * in the datasheet, and the English ones have been checked against a
 * real SAA5050.  Occasionally, different languages have different
 * glyphs for the same character -- these are represented as
 * alternates, with the default being the glyph that looks best.
 *
 * There are some extra glyphs included as well, some derived from
 * standard ones and some made from whole cloth.  They are built on
 * the same 5x9 matrix as the originals, and processed in the same way.
 */

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define XSIZE 6
#define YSIZE 10

/*
 * Design parameters.  These can vary between fonts in the Bedstead family.
 *
 * To faithfully represent the output of an SAA5050, we need to know a
 * little about TV pixel aspect ratios:
 *
 * http://www.lurkertech.com/lg/video-systems/#sqnonsq
 *
 * The SAA5050 can be used to drive both 480i ("NTSC") and 576i
 * ("PAL") displays.  In 480i, industry convention is that you get
 * square pixels with a pixel clock of 12+3/11 MHz.  In 576i, you need
 * 14.75 MHz.  The SAA5050 takes a nominal 6MHz clock and uses both
 * edges to output subpixels (duty cycle between 0.4 and 0.6).  This
 * means that the nominal pixel aspect ratios are 12+3/11:12 for 480i
 * and 14.75:12 for 576i.  These correspond to pixel widths of 102.3
 * and 122.9 respectively.
 *
 * 102.3 is close enough to 100 that we may as well just use square
 * pixels, which will work better on modern displays.  Similarly,
 * 122.9 is close to 125 and keeping the widths as simple fractions
 * makes the fonts easier to work with.
 *
 * Double-height mode requires fonts half the usual width, hence the
 * existence of ultra condensed and extra condensed forms.  The other
 * two widths just fill in the gap with the obvious fractions of the
 * normal width.
 */

static struct width {
	char const * option;
	char const * suffix;
	double xpix;
	int ttfwidth;
} const widths[] = {
	{
		"--normal",
		"",
		100,		/* xpix */
		5,		/* ttfwidth */
	},
	{
		"--ultra-condensed",
		" Ultra Condensed",
		50,		/* xpix */
		1,		/* ttfwidth */
	},
	{
		"--extra-condensed",
		" Extra Condensed",
		62.5,		/* xpix */
		2,		/* ttfwidth */
	},
	{
		"--condensed",
		" Condensed",
		75,		/* xpix */
		3,		/* ttfwidth */
	},
	{
		"--semi-condensed",
		" Semi Condensed",
		87.5,		/* xpix */
		4,		/* ttfwidth */
	},
	{
		"--extended",
		" Extended",
		125,		/* xpix */
		7,		/* ttfwidth */
	},
};
static int const nwidths = sizeof(widths) / sizeof(widths[0]);

struct width const *width = &widths[0];

static struct weight {
	char const *option;
	char const *suffix;
	int weight; /* Expressed in internal units */
	int ttfweight;
} const weights[] = {
	{
		"--medium",
		"",
		0,
		500,
	},
	{
		"--book",
		" Book",
		-25,
		400,
	},
};

static int const nweights = sizeof(weights) / sizeof(weights[0]);

struct weight const *weight = &weights[0];

/* Size of output pixels in font design units (usually 1000/em) */
#define XPIX (width->xpix)
#define YPIX 100

/* Internally, we work in pixels 100 design units square */
#define XPIX_S 100
#define YPIX_S 100

#define XSCALE ((double)XPIX_S / (double)XPIX)
#define YSCALE ((double)YPIX_S / (double)YPIX)

/* Position of diagonal lines within pixels */
/* 29 is approximately 100 * (1-1/sqrt(2)) */
#define XQTR_S 29
#define YQTR_S 29

static void dochar(char const data[YSIZE], unsigned flags);
static void dochar_plotter(char const data[YSIZE], unsigned flags);
static void domosaic(unsigned code, bool sep);
static void domosaic4(unsigned code, bool sep);
static void doalias(char const *alias_of);
static void dopanose(void);
static void glyph_complement(void);

/* U(N) sets the code point and name of a glyph not in AGLFN */
#define U(N) 0x ## N, 0x ## N >= 0x10000 ? "u" #N : "uni" #N
#define ALIAS(alias, canonical) {{},-1,alias,0,canonical}

static struct glyph {
	char data[YSIZE];
	int unicode;
	char const *name;
	unsigned int flags;
#define SEP  0x01 /* Separated graphics */
#define MOS6 0x02 /* 6-cell mosaic graphics character */
#define MOS4 0x04 /* 4-cell mosaic graphics character */
#define SEP6 (SEP | MOS6)
#define SEP4 (SEP | MOS4)
	char const *alias_of;
} const glyphs[] = {
 /*
  * The first batch of glyphs comes from the code tables at the end of
  * the Mullard SAA5050 series datasheet, dated July 1982.
  */
 /* US ASCII (SAA5055) character set */
 {{000,000,000,000,000,000,000,000,000}, 0x0020, "space" },
 {{004,004,004,004,004,000,004,000,000}, 0x0021, "exclam" },
 {{012,012,012,000,000,000,000,000,000}, 0x0022, "quotedbl" },
 {{012,012,037,012,037,012,012,000,000}, 0x0023, "numbersign" },
 {{016,025,024,016,005,025,016,000,000}, 0x0024, "dollar" },
 {{030,031,002,004,010,023,003,000,000}, 0x0025, "percent" },
 {{010,024,024,010,025,022,015,000,000}, 0x0026, "ampersand" },
 {{004,004,010,000,000,000,000,000,000}, 0x2019, "quoteright" },
 {{002,004,010,010,010,004,002,000,000}, 0x0028, "parenleft" },
 {{010,004,002,002,002,004,010,000,000}, 0x0029, "parenright" },
 {{004,025,016,004,016,025,004,000,000}, 0x002a, "asterisk" },
 {{000,004,004,037,004,004,000,000,000}, 0x002b, "plus" },
 {{000,000,000,000,000,004,004,010,000}, 0x002c, "comma"},
 {{000,000,000,016,000,000,000,000,000}, 0x002d, "hyphen" },
 {{000,000,000,000,000,000,004,000,000}, 0x002e, "period"},
 {{000,001,002,004,010,020,000,000,000}, 0x002f, "slash" },
 {{004,012,021,021,021,012,004,000,000}, 0x0030, "zero" },
 {{004,014,004,004,004,004,016,000,000}, 0x0031, "one" },
 {{016,021,001,006,010,020,037,000,000}, 0x0032, "two" },
 {{037,001,002,006,001,021,016,000,000}, 0x0033, "three" },
 {{002,006,012,022,037,002,002,000,000}, 0x0034, "four" },
 {{037,020,036,001,001,021,016,000,000}, 0x0035, "five" },
 {{006,010,020,036,021,021,016,000,000}, 0x0036, "six" },
 {{037,001,002,004,010,010,010,000,000}, 0x0037, "seven" },
 {{016,021,021,016,021,021,016,000,000}, 0x0038, "eight" },
 {{016,021,021,017,001,002,014,000,000}, 0x0039, "nine" },
 {{000,000,004,000,000,000,004,000,000}, 0x003a, "colon"},
 {{000,000,004,000,000,004,004,010,000}, 0x003b, "semicolon"},
 {{002,004,010,020,010,004,002,000,000}, 0x003c, "less" },
 {{000,000,037,000,037,000,000,000,000}, 0x003d, "equal" },
 {{010,004,002,001,002,004,010,000,000}, 0x003e, "greater" },
 {{016,021,002,004,004,000,004,000,000}, 0x003f, "question" },
 {{016,021,027,025,027,020,016,000,000}, 0x0040, "at" },
 {{004,012,021,021,037,021,021,000,000}, 0x0041, "A" },
 {{036,021,021,036,021,021,036,000,000}, 0x0042, "B" },
 {{016,021,020,020,020,021,016,000,000}, 0x0043, "C" },
 {{036,021,021,021,021,021,036,000,000}, 0x0044, "D" },
 {{037,020,020,036,020,020,037,000,000}, 0x0045, "E" },
 {{037,020,020,036,020,020,020,000,000}, 0x0046, "F" },
 {{016,021,020,020,023,021,017,000,000}, 0x0047, "G" },
 {{021,021,021,037,021,021,021,000,000}, 0x0048, "H" },
 {{016,004,004,004,004,004,016,000,000}, 0x0049, "I" },
 {{001,001,001,001,001,021,016,000,000}, 0x004a, "J" },
 {{021,022,024,030,024,022,021,000,000}, 0x004b, "K" },
 {{020,020,020,020,020,020,037,000,000}, 0x004c, "L" },
 {{021,033,025,025,021,021,021,000,000}, 0x004d, "M" },
 {{021,021,031,025,023,021,021,000,000}, 0x004e, "N" },
 {{016,021,021,021,021,021,016,000,000}, 0x004f, "O" },
 {{036,021,021,036,020,020,020,000,000}, 0x0050, "P" },
 {{016,021,021,021,025,022,015,000,000}, 0x0051, "Q" },
 {{036,021,021,036,024,022,021,000,000}, 0x0052, "R" },
 {{016,021,020,016,001,021,016,000,000}, 0x0053, "S" },
 {{037,004,004,004,004,004,004,000,000}, 0x0054, "T" },
 {{021,021,021,021,021,021,016,000,000}, 0x0055, "U" },
 {{021,021,021,012,012,004,004,000,000}, 0x0056, "V" },
 {{021,021,021,025,025,025,012,000,000}, 0x0057, "W" },
 {{021,021,012,004,012,021,021,000,000}, 0x0058, "X" },
 {{021,021,012,004,004,004,004,000,000}, 0x0059, "Y" },
 {{037,001,002,004,010,020,037,000,000}, 0x005a, "Z" },
 {{017,010,010,010,010,010,017,000,000}, 0x005b, "bracketleft" },
 {{000,020,010,004,002,001,000,000,000}, 0x005c, "backslash" },
 {{036,002,002,002,002,002,036,000,000}, 0x005d, "bracketright" },
 {{004,012,021,000,000,000,000,000,000}, 0x005e, "asciicircum" },
 {{000,000,000,000,000,000,037,000,000}, 0x005f, "underscore" },
 {{004,004,002,000,000,000,000,000,000}, 0x201b, "quotereversed" },
 {{000,000,016,001,017,021,017,000,000}, 0x0061, "a" },
 {{020,020,036,021,021,021,036,000,000}, 0x0062, "b" },
 {{000,000,017,020,020,020,017,000,000}, 0x0063, "c" },
 {{001,001,017,021,021,021,017,000,000}, 0x0064, "d" },
 {{000,000,016,021,037,020,016,000,000}, 0x0065, "e" },
 {{002,004,004,016,004,004,004,000,000}, 0x0066, "f" },
 {{000,000,017,021,021,021,017,001,016}, 0x0067, "g" },
 {{020,020,036,021,021,021,021,000,000}, 0x0068, "h" },
 {{004,000,014,004,004,004,016,000,000}, 0x0069, "i" },
 {{004,000,004,004,004,004,004,004,010}, 0x006a, "j" },
 {{010,010,011,012,014,012,011,000,000}, 0x006b, "k" },
 {{014,004,004,004,004,004,016,000,000}, 0x006c, "l" },
 {{000,000,032,025,025,025,025,000,000}, 0x006d, "m" },
 {{000,000,036,021,021,021,021,000,000}, 0x006e, "n" },
 {{000,000,016,021,021,021,016,000,000}, 0x006f, "o" },
 {{000,000,036,021,021,021,036,020,020}, 0x0070, "p" },
 {{000,000,017,021,021,021,017,001,001}, 0x0071, "q" },
 {{000,000,013,014,010,010,010,000,000}, 0x0072, "r" },
 {{000,000,017,020,016,001,036,000,000}, 0x0073, "s" },
 {{004,004,016,004,004,004,002,000,000}, 0x0074, "t" },
 {{000,000,021,021,021,021,017,000,000}, 0x0075, "u" },
 {{000,000,021,021,012,012,004,000,000}, 0x0076, "v" },
 {{000,000,021,021,025,025,012,000,000}, 0x0077, "w" },
 {{000,000,021,012,004,012,021,000,000}, 0x0078, "x" },
 {{000,000,021,021,021,021,017,001,016}, 0x0079, "y" },
 {{000,000,037,002,004,010,037,000,000}, 0x007a, "z" },
 {{003,004,004,010,004,004,003,000,000}, 0x007b, "braceleft" },
 {{004,004,004,000,004,004,004,000,000}, 0x00a6, "brokenbar" },
 {{030,004,004,002,004,004,030,000,000}, 0x007d, "braceright" },
 {{010,025,002,000,000,000,000,000,000}, 0x007e, "asciitilde" },
 {{037,037,037,037,037,037,037,000,000}, 0x25a0, "filledbox" },

 /* Extra characters found in the English (SAA5050) character set */
 {{006,011,010,034,010,010,037,000,000}, 0x00a3, "sterling" },
 {{004,004,004,000,000,000,000,000,000}, 0x0027, "quotesingle" },
 {{000,004,010,037,010,004,000,000,000}, 0x2190, "arrowleft" },
 {{020,020,020,020,026,001,002,004,007}, 0x00bd, "onehalf" },
 {{000,004,002,037,002,004,000,000,000}, 0x2192, "arrowright" },
 {{000,004,016,025,004,004,000,000,000}, 0x2191, "arrowup" },
 {{000,000,000,037,000,000,000,000,000}, 0x2014, "emdash" },
 {{010,010,010,010,011,003,005,007,001}, 0x00bc, "onequarter" },
 {{012,012,012,012,012,012,012,000,000}, 0x2016, "dblverticalbar" },
 {{030,004,030,004,031,003,005,007,001}, 0x00be, "threequarters" },
 {{000,004,000,037,000,004,000,000,000}, 0x00f7, "divide" },

 /* Extra characters found in the German (SAA5051) character set */
 {{000,000,000,000,000,010,010,020,000}, -1, "comma.saa5051" },
 {{000,000,000,000,000,014,014,000,000}, -1, "period.saa5051" },
 {{000,000,000,010,000,000,010,000,000}, -1, "colon.saa5051" },
 {{000,000,010,000,000,010,010,020,000}, -1, "semicolon.saa5051" },
 {{016,021,020,016,021,016,001,021,016}, 0x00a7, "section" },
 {{012,000,016,021,037,021,021,000,000}, 0x00c4, "Adieresis" },
 {{012,000,016,021,021,021,016,000,000}, 0x00d6, "Odieresis" },
 {{012,000,021,021,021,021,016,000,000}, 0x00dc, "Udieresis" },
 {{006,011,006,000,000,000,000,000,000}, 0x00b0, "degree" },
 {{012,000,016,001,017,021,017,000,000}, 0x00e4, "adieresis" },
 {{000,012,000,016,021,021,016,000,000}, 0x00f6, "odieresis" },
 {{000,012,000,021,021,021,017,000,000}, 0x00fc, "udieresis" },
 {{014,022,022,026,021,021,026,020,020}, 0x00df, "germandbls" },

 /* Extra characters found in the Swedish (SAA5052) character set */
 {{000,000,021,016,012,016,021,000,000}, 0x00a4, "currency" },
 {{002,004,037,020,036,020,037,000,000}, 0x00c9, "Eacute" },
 {{016,011,011,011,011,011,016,000,000}, -1, "D.saa5052" },
 {{010,010,010,010,010,010,017,000,000}, -1, "L.saa5052" },
 {{004,000,016,021,037,021,021,000,000}, 0x00c5, "Aring" },
 {{002,004,016,021,037,020,016,000,000}, 0x00e9, "eacute" },
 {{004,000,016,001,017,021,017,000,000}, 0x00e5, "aring" },

 /* Extra characters found in the Italian (SAA5053) character set */
 {{000,000,017,020,020,020,017,002,004}, 0x00e7, "ccedilla" },
 {{010,004,021,021,021,021,017,000,000}, 0x00f9, "ugrave" },
 {{010,004,016,001,017,021,017,000,000}, 0x00e0, "agrave" },
 {{010,004,000,016,021,021,016,000,000}, 0x00f2, "ograve" },
 {{010,004,016,021,037,020,016,000,000}, 0x00e8, "egrave" },
 {{010,004,000,014,004,004,016,000,000}, 0x00ec, "igrave" },

 /* Extra characters found in the Belgian (SAA5054) character set */
 {{012,000,014,004,004,004,016,000,000}, 0x00ef, "idieresis" },
 {{012,000,016,021,037,020,016,000,000}, 0x00eb, "edieresis" },
 {{004,012,016,021,037,020,016,000,000}, 0x00ea, "ecircumflex" },
 {{004,002,021,021,021,021,017,000,000}, -1, "ugrave.saa5054" },
 {{004,012,000,014,004,004,016,000,000}, 0x00ee, "icircumflex" },
 {{004,012,016,001,017,021,017,000,000}, 0x00e2, "acircumflex" },
 {{004,012,016,021,021,021,016,000,000}, -1, "ocircumflex.saa5054" },
 {{004,012,000,021,021,021,017,000,000}, 0x00fb, "ucircumflex" },
 {{000,000,017,020,020,020,017,002,006}, -1, "ccedilla.saa5054" },

 /* Extra characters found in the Hebrew (SAA5056) character set */
 {{000,021,011,025,022,021,021,000,000}, U(05D0) }, /* alef */
 {{000,016,002,002,002,002,037,000,000}, U(05D1) }, /* bet */
 {{000,003,001,001,003,005,011,000,000}, U(05D2) }, /* gimel */
 {{000,037,002,002,002,002,002,000,000}, U(05D3) }, /* dalet */
 {{000,037,001,001,021,021,021,000,000}, U(05D4) }, /* he */
 {{000,014,004,004,004,004,004,000,000}, U(05D5) }, /* vav */
 {{000,016,004,010,004,004,004,000,000}, U(05D6) }, /* zayin */
 {{000,037,021,021,021,021,021,000,000}, U(05D7) }, /* het */
 {{000,021,023,025,021,021,037,000,000}, U(05D8) }, /* tet */
 {{000,014,004,000,000,000,000,000,000}, U(05D9) }, /* yod */
 {{000,037,001,001,001,001,001,001,000}, U(05DA) }, /* kaffinal */
 {{000,037,001,001,001,001,037,000,000}, U(05DB) }, /* kaf */
 {{020,037,001,001,001,002,014,000,000}, U(05DC) }, /* lamed */
 {{000,037,021,021,021,021,037,000,000}, U(05DD) }, /* memfinal */
 {{000,026,011,021,021,021,027,000,000}, U(05DE) }, /* mem */
 {{000,014,004,004,004,004,004,004,004}, U(05DF) }, /* nunfinal */
 {{000,006,002,002,002,002,016,000,000}, U(05E0) }, /* nun */
 {{000,037,011,021,021,021,016,000,000}, U(05E1) }, /* samekh */
 {{000,011,011,011,011,012,034,000,000}, U(05E2) }, /* ayin */
 {{000,037,011,015,001,001,001,001,000}, U(05E3) }, /* pefinal */
 {{000,037,011,015,001,001,037,000,000}, U(05E4) }, /* pe */
 {{000,031,012,014,010,010,010,010,000}, U(05E5) }, /* tsadifinal */
 {{000,021,021,012,004,002,037,000,000}, U(05E6) }, /* tsadi */
 {{000,037,001,011,011,012,010,010,000}, U(05E7) }, /* qof */
 {{000,037,001,001,001,001,001,000,000}, U(05E8) }, /* resh */
 {{000,025,025,025,031,021,036,000,000}, U(05E9) }, /* shin */
 {{000,017,011,011,011,011,031,000,000}, U(05EA) }, /* tav */
 {{000,000,025,025,016,000,000,000,000}, -1, "oldsheqel" },

 /* Extra characters found in the Cyrillic (SAA5057) character set */
 {{000,000,021,021,035,025,035,000,000}, U(044B) }, /* yeru */
 {{022,025,025,035,025,025,022,000,000}, U(042E) }, /* Iu */
 {{016,021,021,021,037,021,021,000,000}, U(0410) }, /* A */
 {{037,020,020,037,021,021,037,000,000}, U(0411) }, /* Be */
 {{022,022,022,022,022,022,037,001,000}, U(0426) }, /* Tse */
 {{006,012,012,012,012,012,037,021,000}, U(0414) }, /* De */
 {{037,020,020,036,020,020,037,000,000}, U(0415) }, /* Ie */
 {{004,037,025,025,025,037,004,000,000}, U(0424) }, /* Ef */
 {{037,020,020,020,020,020,020,000,000}, U(0413) }, /* Ghe */
 {{021,021,012,004,012,021,021,000,000}, U(0425) }, /* Ha */
 {{021,021,023,025,031,021,021,000,000}, U(0418) }, /* I */
 {{025,021,023,025,031,021,021,000,000}, U(0419) }, /* Ishort */
 {{021,022,024,030,024,022,021,000,000}, U(041A) }, /* Ka */
 {{007,011,011,011,011,011,031,000,000}, U(041B) }, /* El */
 {{021,033,025,025,021,021,021,000,000}, U(041C) }, /* Em */
 {{021,021,021,037,021,021,021,000,000}, U(041D) }, /* En */
 {{016,021,021,021,021,021,016,000,000}, U(041E) }, /* O */
 {{037,021,021,021,021,021,021,000,000}, U(041F) }, /* Pe */
 {{017,021,021,017,005,011,021,000,000}, U(042F) }, /* Ya */
 {{036,021,021,036,020,020,020,000,000}, U(0420) }, /* Er */
 {{016,021,020,020,020,021,016,000,000}, U(0421) }, /* Es */
 {{037,004,004,004,004,004,004,000,000}, U(0422) }, /* Te */
 {{021,021,021,037,001,001,037,000,000}, U(0423) }, /* U */
 {{025,025,025,016,025,025,025,000,000}, U(0416) }, /* Zhe */
 {{036,021,021,036,021,021,036,000,000}, U(0412) }, /* Ve */
 {{020,020,020,037,021,021,037,000,000}, U(042C) }, /* Soft */
 {{030,010,010,017,011,011,017,000,000}, U(042A) }, /* Hard */
 {{016,021,001,006,001,021,016,000,000}, U(0417) }, /* Ze */
 {{025,025,025,025,025,025,037,000,000}, U(0428) }, /* Sha */
 {{014,022,001,007,001,022,014,000,000}, U(042D) }, /* E */
 {{025,025,025,025,025,025,037,001,000}, U(0429) }, /* Shcha */
 {{021,021,021,037,001,001,001,000,000}, U(0427) }, /* Che */
 {{021,021,021,035,025,025,035,000,000}, U(042B) }, /* Yeru */
 {{000,000,022,025,035,025,022,000,000}, U(044E) }, /* yu */
 {{000,000,016,001,017,021,017,000,000}, U(0430) }, /* a */
 {{016,020,036,021,021,021,036,000,000}, U(0431) }, /* be */
 {{000,000,022,022,022,022,037,001,000}, U(0446) }, /* tse */
 {{000,000,006,012,012,012,037,021,000}, U(0434) }, /* de */
 {{000,000,016,021,037,020,016,000,000}, U(0435) }, /* ie */
 {{000,004,016,025,025,025,016,004,000}, U(0444) }, /* ef */
 {{000,000,037,020,020,020,020,000,000}, U(0433) }, /* ghe */
 {{000,000,021,012,004,012,021,000,000}, U(0445) }, /* ha */
 {{000,000,021,023,025,031,021,000,000}, U(0438) }, /* i */
 {{000,004,021,023,025,031,021,000,000}, U(0439) }, /* ishort */
 {{000,000,021,022,034,022,021,000,000}, U(043A) }, /* ka */
 {{000,000,007,011,011,011,031,000,000}, U(043B) }, /* el */
 {{000,000,021,033,025,021,021,000,000}, U(043C) }, /* em */
 {{000,000,021,021,037,021,021,000,000}, U(043D) }, /* en */
 {{000,000,016,021,021,021,016,000,000}, U(043E) }, /* o */
 {{000,000,037,021,021,021,021,000,000}, U(043F) }, /* pe */
 {{000,000,017,021,017,005,031,000,000}, U(044F) }, /* ya */
 {{000,000,036,021,021,021,036,020,020}, U(0440) }, /* er */
 {{000,000,016,021,020,021,016,000,000}, U(0441) }, /* es */
 {{000,000,037,004,004,004,004,000,000}, U(0442) }, /* te */
 {{000,000,021,021,021,021,017,001,016}, U(0443) }, /* u */
 {{000,000,025,025,016,025,025,000,000}, U(0436) }, /* zhe */
 {{000,000,036,021,036,021,036,000,000}, U(0432) }, /* ve */
 {{000,000,020,020,036,021,036,000,000}, U(044C) }, /* soft */
 {{000,000,030,010,016,011,016,000,000}, U(044A) }, /* hard */
 {{000,000,016,021,006,021,016,000,000}, U(0437) }, /* ze */
 {{000,000,025,025,025,025,037,000,000}, U(0448) }, /* sha */
 {{000,000,014,022,006,022,014,000,000}, U(044D) }, /* e */
 {{000,000,025,025,025,025,037,001,000}, U(0449) }, /* shcha */
 {{000,000,021,021,021,017,001,000,000}, U(0447) }, /* che */

 /*
  * The other batch of glyphs were specially designed for this font.
  * These are kept sorted by Unicode code point.
  */

 /* Basic Latin */
 {{010,004,002,000,000,000,000,000,000}, 0x0060, "grave" },
 {{004,004,004,004,004,004,004,000,000}, 0x007c, "bar" },

 /* Latin-1 supplement */
 {{000,000,000,000,000,000,000,000,000}, U(00A0) }, /* non-breaking space */
 {{000,000,004,000,004,004,004,004,004}, 0x00a1, "exclamdown" },
 {{000,004,017,024,024,024,017,004,000}, 0x00a2, "cent" },
 {{021,012,037,004,037,004,004,000,000}, 0x00a5, "yen" },
 {{012,000,000,000,000,000,000,000,000}, 0x00a8, "dieresis" },
 {{016,021,025,033,031,033,025,021,016}, 0x00a9, "copyright" },
 {{016,001,017,021,017,000,037,000,000}, 0x00aa, "ordfeminine" },
 {{000,000,000,011,022,011,000,000,000}, 0x00ab, "guillemotleft" },
 {{000,000,037,001,001,000,000,000,000}, 0x00ac, "logicalnot" },
 {{016,021,035,033,033,035,033,021,016}, 0x00ae, "registered" },
 {{016,000,000,000,000,000,000,000,000}, 0x00af, "macron" },
 {{004,004,037,004,004,000,037,000,000}, 0x00b1, "plusminus" },
 {{014,002,004,010,016,000,000,000,000}, 0x00b2, "twosuperior" },
 {{014,002,014,002,014,000,000,000,000}, 0x00b3, "threesuperior" },
 {{002,004,010,000,000,000,000,000,000}, 0x00b4, "acute" },
 {{000,000,022,022,022,022,035,020,020}, 0x00b5, "mu" },
 {{015,025,025,015,005,005,005,000,000}, 0x00b6, "paragraph" },
 {{000,000,000,004,000,000,000,000,000}, 0x00b7, "periodcentered" },
 {{000,000,000,000,000,000,004,002,004}, 0x00b8, "cedilla" },
 {{004,014,004,004,016,000,000,000,000}, 0x00b9, "onesuperior" },
 {{016,021,021,021,016,000,037,000,000}, 0x00ba, "ordmasculine" },
 {{000,000,000,022,011,022,000,000,000}, 0x00bb, "guillemotright" },
 {{000,000,004,000,004,004,010,021,016}, 0x00bf, "questiondown" },
 {{010,004,016,021,037,021,021,000,000}, 0x00c0, "Agrave" },
 {{002,004,016,021,037,021,021,000,000}, 0x00c1, "Aacute" },
 {{004,012,016,021,037,021,021,000,000}, 0x00c2, "Acircumflex" },
 {{005,012,016,021,037,021,021,000,000}, 0x00c3, "Atilde" },
 {{017,024,024,026,034,024,027,000,000}, 0x00c6, "AE" },
 {{016,021,020,020,020,021,016,004,010}, 0x00c7, "Ccedilla" },
 {{010,004,037,020,036,020,037,000,000}, 0x00c8, "Egrave" },
 {{004,012,037,020,036,020,037,000,000}, 0x00ca, "Ecircumflex" },
 {{012,000,037,020,036,020,037,000,000}, 0x00cb, "Edieresis" },
 {{010,004,000,016,004,004,016,000,000}, 0x00cc, "Igrave" },
 {{002,004,000,016,004,004,016,000,000}, 0x00cd, "Iacute" },
 {{004,012,000,016,004,004,016,000,000}, 0x00ce, "Icircumflex" },
 {{012,000,016,004,004,004,016,000,000}, 0x00cf, "Idieresis" },
 {{016,011,011,035,011,011,016,000,000}, 0x00d0, "Eth" },
 {{005,012,000,031,025,023,021,000,000}, 0x00d1, "Ntilde" },
 {{010,004,016,021,021,021,016,000,000}, 0x00d2, "Ograve" },
 {{002,004,016,021,021,021,016,000,000}, 0x00d3, "Oacute" },
 {{004,012,016,021,021,021,016,000,000}, 0x00d4, "Ocircumflex" },
 {{005,012,016,021,021,021,016,000,000}, 0x00d5, "Otilde" },
 {{000,021,012,004,012,021,000,000,000}, 0x00d7, "multiply" },
 {{016,021,023,025,031,021,016,000,000}, 0x00d8, "Oslash" },
 {{010,004,021,021,021,021,016,000,000}, 0x00d9, "Ugrave" },
 {{002,004,021,021,021,021,016,000,000}, 0x00da, "Uacute" },
 {{004,012,000,021,021,021,016,000,000}, 0x00db, "Ucircumflex" },
 {{002,004,021,012,004,004,004,000,000}, 0x00dd, "Yacute" },
 {{020,036,021,021,036,020,020,000,000}, 0x00de, "Thorn" },
 {{002,004,016,001,017,021,017,000,000}, 0x00e1, "aacute" },
 {{005,012,016,001,017,021,017,000,000}, 0x00e3, "atilde" },
 {{000,000,012,005,017,024,016,000,000}, 0x00e6, "ae" },
 {{002,004,000,014,004,004,016,000,000}, 0x00ed, "iacute" },
 {{032,004,012,001,017,021,016,000,000}, 0x00f0, "eth" },
 {{005,012,036,021,021,021,021,000,000}, 0x00f1, "ntilde" },
 {{002,004,000,016,021,021,016,000,000}, 0x00f3, "oacute" },
 {{004,012,000,016,021,021,016,000,000}, 0x00f4, "ocircumflex" },
 {{005,012,000,016,021,021,016,000,000}, 0x00f5, "otilde" },
 {{000,000,015,022,025,011,026,000,000}, 0x00f8, "oslash" },
 {{002,004,021,021,021,021,017,000,000}, 0x00fa, "uacute" },
 {{002,004,021,021,021,021,017,001,016}, 0x00fd, "yacute" },
 {{020,020,036,021,021,021,036,020,020}, 0x00fe, "thorn" },
 {{012,000,021,021,021,021,017,001,016}, 0x00ff, "ydieresis" },

 /* Latin extended-A */
 {{016,000,016,021,037,021,021,000,000}, 0x0100, "Amacron" },
 {{016,000,016,001,017,021,017,000,000}, 0x0101, "amacron" },
 {{004,012,021,021,037,021,021,002,001}, 0x0104, "Aogonek" },
 {{000,000,016,001,017,021,017,002,001}, 0x0105, "aogonek" },
 {{002,004,016,021,020,021,016,000,000}, 0x0106, "Cacute" },
 {{002,004,017,020,020,020,017,000,000}, 0x0107, "cacute" },
 {{004,012,016,021,020,021,016,000,000}, 0x0108, "Ccircumflex" },
 {{004,012,017,020,020,020,017,000,000}, 0x0109, "ccircumflex" },
 {{004,000,016,021,020,021,016,000,000}, 0x010a, "Cdotaccent" },
 {{004,000,017,020,020,020,017,000,000}, 0x010b, "cdotaccent" },
 {{012,004,016,021,020,021,016,000,000}, 0x010c, "Ccaron" },
 {{012,004,017,020,020,020,017,000,000}, 0x010d, "ccaron" },
 {{012,004,036,021,021,021,036,000,000}, 0x010e, "Dcaron" },
 {{005,005,014,024,024,024,014,000,000}, 0x010f, "dcaron" },
 {{016,011,011,035,011,011,016,000,000}, 0x0110, "Dcroat" },
 {{002,007,002,016,022,022,016,000,000}, 0x0111, "dcroat" },
 {{016,000,037,020,036,020,037,000,000}, 0x0112, "Emacron" },
 {{016,000,016,021,037,020,016,000,000}, 0x0113, "emacron" },
 {{004,000,037,020,036,020,037,000,000}, 0x0116, "Edotaccent" },
 {{004,000,016,021,037,020,016,000,000}, 0x0117, "edotaccent" },
 {{037,020,020,036,020,020,037,002,001}, 0x0118, "Eogonek" },
 {{000,000,016,021,037,020,016,002,001}, 0x0119, "eogonek" },
 {{012,004,037,020,036,020,037,000,000}, 0x011a, "Ecaron" },
 {{012,004,016,021,037,020,016,000,000}, 0x011b, "ecaron" },
 {{004,012,017,020,023,021,017,000,000}, 0x011c, "Gcircumflex" },
 {{004,012,017,021,021,021,017,001,016}, 0x011d, "gcircumflex" },
 {{004,000,017,020,023,021,017,000,000}, 0x0120, "Gdotaccent" },
 {{004,000,017,021,021,021,017,001,016}, 0x0121, "gdotaccent" },
 {{021,037,021,037,021,021,021,000,000}, 0x0126, "Hbar" },
 {{010,034,010,016,011,011,011,000,000}, 0x0127, "hbar" },
 {{005,012,000,016,004,004,016,000,000}, 0x0128, "Itilde" },
 {{005,012,000,014,004,004,016,000,000}, 0x0129, "itilde" },
 {{016,000,016,004,004,004,016,000,000}, 0x012a, "Imacron" },
 {{016,000,014,004,004,004,016,000,000}, 0x012b, "imacron" },
 {{016,004,004,004,004,004,016,004,002}, 0x012e, "Iogonek" },
 {{004,000,014,004,004,004,016,004,002}, 0x012f, "iogonek" },
 {{004,000,016,004,004,004,016,000,000}, 0x0130, "Idotaccent" },
 {{000,000,014,004,004,004,016,000,000}, 0x0131, "dotlessi" },
 {{021,021,021,021,021,025,022,000,000}, 0x0132, "IJ" },
 {{011,000,031,011,011,011,035,001,002}, 0x0133, "ij" },
 {{002,005,000,002,002,022,014,000,000}, 0x0134, "Jcircumflex" },
 {{004,012,000,004,004,004,004,004,010}, 0x0135, "jcircumflex" },
 {{000,000,021,022,034,022,021,000,000}, 0x0138, "kgreenlandic" },
 {{022,022,024,020,020,020,037,000,000}, 0x013d, "Lcaron" },
 {{031,011,012,010,010,010,034,000,000}, 0x013e, "lcaron" },
 {{020,020,020,021,020,020,037,000,000}, 0x013f, "Ldot" },
 {{030,010,010,011,010,010,034,000,000}, 0x0140, "ldot" },
 {{010,010,014,030,010,010,017,000,000}, 0x0141, "Lslash" },
 {{014,004,006,014,004,004,016,000,000}, 0x0142, "lslash" },
 {{002,004,021,031,025,023,021,000,000}, 0x0143, "Nacute" },
 {{002,004,036,021,021,021,021,000,000}, 0x0144, "nacute" },
 {{012,004,021,031,025,023,021,000,000}, 0x0147, "Ncaron" },
 {{012,004,036,021,021,021,021,000,000}, 0x0148, "ncaron" },
 {{020,020,026,005,005,005,005,000,000}, 0x0149, "napostrophe" },
 {{021,021,031,025,023,021,021,001,016}, 0x014a, "Eng" },
 {{000,000,036,021,021,021,021,001,016}, 0x014b, "eng" },
 {{016,000,016,021,021,021,016,000,000}, 0x014c, "Omacron" },
 {{000,016,000,016,021,021,016,000,000}, 0x014d, "omacron" },
 {{017,024,024,026,024,024,017,000,000}, 0x0152, "OE" },
 {{000,000,012,025,027,024,013,000,000}, 0x0153, "oe" },
 {{002,004,036,021,036,022,021,000,000}, 0x0154, "Racute" },
 {{001,002,013,014,010,010,010,000,000}, 0x0155, "racute" },
 {{012,004,036,021,036,022,021,000,000}, 0x0158, "Rcaron" },
 {{005,002,013,014,010,010,010,000,000}, 0x0159, "rcaron" },
 {{002,004,017,020,016,001,036,000,000}, 0x015b, "sacute" },
 {{016,021,020,016,001,021,016,004,010}, 0x015e, "Scedilla" },
 {{000,000,017,020,016,001,036,004,010}, 0x015f, "scedilla" },
 {{012,004,017,020,016,001,036,000,000}, 0x0161, "scaron" },
 {{037,004,004,004,004,004,004,002,004}, U(0162) }, /* Tcedilla */
 {{004,004,016,004,004,005,002,002,004}, U(0163) }, /* tcedilla */
 {{012,004,037,004,004,004,004,000,000}, 0x0164, "Tcaron" },
 {{011,011,034,010,010,010,004,000,000}, 0x0165, "tcaron" },
 {{016,000,021,021,021,021,016,000,000}, 0x016a, "Umacron" },
 {{000,016,000,021,021,021,017,000,000}, 0x016b, "umacron" },
 {{004,000,021,021,021,021,016,000,000}, 0x016e, "Uring" },
 {{004,000,021,021,021,021,017,000,000}, 0x016f, "uring" },
 {{021,021,021,021,021,021,016,004,002}, 0x0172, "Uogonek" },
 {{000,000,021,021,021,021,017,002,001}, 0x0173, "uogonek" },
 {{004,012,000,021,025,025,012,000,000}, 0x0174, "Wcircumflex" },
 {{004,012,000,021,021,025,012,000,000}, 0x0175, "wcircumflex" },
 {{004,012,000,021,012,004,004,000,000}, 0x0176, "Ycircumflex" },
 {{004,012,000,021,021,021,017,001,016}, 0x0177, "ycircumflex" },
 {{012,000,021,012,004,004,004,000,000}, 0x0178, "Ydieresis" },
 {{002,004,037,001,016,020,037,000,000}, 0x0179, "Zacute" },
 {{002,004,037,002,004,010,037,000,000}, 0x017a, "zacute" },
 {{004,000,037,001,016,020,037,000,000}, 0x017b, "Zdotaccent" },
 {{004,000,037,002,004,010,037,000,000}, 0x017c, "zdotaccent" },
 {{012,004,037,001,016,020,037,000,000}, 0x017d, "Zcaron" },
 {{012,004,037,002,004,010,037,000,000}, 0x017e, "zcaron" },
 {{002,004,004,004,004,004,004,000,000}, 0x017f, "longs" },

 /* Latin extended-B */
 {{003,002,017,020,020,020,017,000,000}, U(0188) }, /* Hooktop C */
 {{037,001,001,017,001,001,037,000,000}, U(018E) }, /* reversed E */
 {{016,021,001,037,021,021,016,000,000}, U(018F) }, /* Schwa */
 {{002,004,004,016,004,004,004,004,010}, 0x0192, "florin" },
 {{006,010,011,012,014,012,011,000,000}, U(0199) }, /* Hooktop K */
 {{030,004,016,004,012,012,021,000,000}, U(019B) }, /* Barred lambda */
 {{000,000,036,021,021,021,021,001,001}, U(019E) }, /* N, right leg */
 {{014,020,036,021,021,021,036,020,020}, U(01A5) }, /* Hooktop P */
 {{016,021,001,016,020,021,016,000,000}, U(01A7) }, /* Tone 2 (reversed S) */
 {{000,000,016,001,016,020,016,000,000}, U(01A8) }, /* tone 2 (reversed s) */
 {{004,004,016,004,004,004,002,014,000}, U(01AB) }, /* Left-hook T */
 {{002,004,016,004,004,004,002,000,000}, U(01AD) }, /* Hooktop T */
 {{016,021,005,002,015,020,037,000,000}, U(01BB) }, /* Barred two */
 {{004,004,004,004,004,004,004,004,004}, U(01C0) }, /* Pipe */
 {{012,012,012,012,012,012,012,012,012}, U(01C1) }, /* Double pipe */
 {{004,004,004,037,004,037,004,004,004}, U(01C2) }, /* Double-barred pipe */
 {{004,004,004,004,004,000,004,000,000}, U(01C3) }, /* Exclamation point */
 {{000,000,016,001,037,021,016,000,000}, U(01DD) }, /* turned e */
 {{016,000,017,024,026,034,027,000,000}, U(01E2) }, /* AEmacron */
 {{016,000,012,005,017,024,016,000,000}, U(01E3) }, /* aemacron */
 {{012,004,000,004,004,004,004,004,010}, U(01F0) }, /* J wedge */
 {{016,000,021,012,004,004,004,000,000}, U(0232) }, /* Ymacron */
 {{016,000,021,021,021,021,017,001,016}, U(0233) }, /* ymacron */
 {{000,000,004,004,004,004,004,004,010}, U(0237) }, /* dotlessj */
 {{000,000,017,020,016,001,036,004,003}, U(023F) }, /* s with swash tail */
 {{000,000,037,002,004,010,030,004,003}, U(0240) }, /* z with swash tail */
 {{004,004,012,012,021,021,021,000,000}, U(0245) }, /* turned V */

 /* IPA extensions */
 {{000,000,036,021,036,020,016,000,000}, U(0250) }, /* Turned A */
 {{000,000,017,021,021,023,015,000,000}, U(0251) }, /* Script A */
 {{000,000,026,031,021,021,036,000,000}, U(0252) }, /* Turned script A */
 {{014,020,036,021,021,021,036,000,000}, U(0253) }, /* Hooktop B */
 {{000,000,036,001,001,001,036,000,000}, U(0254) }, /* Open O */
 {{000,000,017,020,022,025,016,004,000}, U(0255) }, /* Curly-tail C */
 {{002,002,016,022,022,022,016,002,001}, U(0256) }, /* Right-tail D */
 {{001,002,016,022,022,022,016,000,000}, U(0257) }, /* Hooktop D */
 {{000,000,016,021,037,001,016,000,000}, U(0258) }, /* Reversed E */
 {{000,000,016,001,037,021,016,000,000}, U(0259) }, /* Schwa */
 {{000,000,030,004,035,026,010,000,000}, U(025A) }, /* Right-hook schwa */
 {{000,000,017,020,016,020,017,000,000}, U(025B) }, /* Epsilon */
 {{000,000,036,001,016,001,036,000,000}, U(025C) }, /* Reversed epsilon */
 {{000,000,030,004,031,006,030,000,000}, U(025D) }, /* Right-hook rev epsilon */
 {{000,000,016,021,026,021,016,000,000}, U(025E) }, /* Closed reversed epsilon */
 {{000,000,004,004,004,016,004,004,010}, U(025F) }, /* Barred dotless J */
 {{001,002,016,022,022,022,016,002,014}, U(0260) }, /* Hooktop G */
 {{000,000,017,021,021,021,017,001,016}, U(0261) }, /* Opentail G */
 {{000,000,017,020,023,021,017,000,000}, U(0262) }, /* Small capital G */
 {{000,000,021,012,012,004,012,012,004}, U(0263) }, /* Gamma */
 {{000,000,033,004,012,012,004,000,000}, U(0264) }, /* Ram's horns */
 {{000,000,021,021,021,021,017,001,001}, U(0265) }, /* Turned H */
 {{014,020,036,021,021,021,021,000,000}, U(0266) }, /* Hooktop H */
 {{014,020,036,021,021,021,021,001,006}, U(0267) }, /* Hooktop heng */
 {{004,000,014,004,016,004,016,000,000}, U(0268) }, /* Barred I */
 {{000,000,014,004,004,004,002,000,000}, U(0269) }, /* Iota */
 {{000,000,016,004,004,004,016,000,000}, U(026A) }, /* Small capital I */
 {{014,004,015,026,004,004,016,000,000}, U(026B) }, /* L with tilde */
 {{014,004,014,025,016,004,016,000,000}, U(026C) }, /* Belted L */
 {{014,004,004,004,004,004,004,004,002}, U(026D) }, /* Right-tail L */
 {{030,010,017,011,012,011,035,001,006}, U(026E) }, /* L-Ezh ligature */
 {{000,000,025,025,025,025,013,000,000}, U(026F) }, /* Turned M */
 {{000,000,025,025,025,025,013,001,001}, U(0270) }, /* Turned M, right tail */
 {{000,000,032,025,025,025,025,001,002}, U(0271) }, /* Left-tail M (at right) */
 {{000,000,016,011,011,011,011,010,020}, U(0272) }, /* Left-tail N (at left) */
 {{000,000,034,022,022,022,022,002,001}, U(0273) }, /* Right-tail N */
 {{000,000,021,031,025,023,021,000,000}, U(0274) }, /* Small capital N */
 {{000,000,016,021,037,021,016,000,000}, U(0275) }, /* Barred O */
 {{000,000,017,024,027,024,017,000,000}, U(0276) }, /* Small capital O-E ligature */
 {{000,000,016,021,025,025,012,000,000}, U(0277) }, /* Closed omega */
 {{004,004,016,025,025,025,016,004,004}, U(0278) }, /* Phi */
 {{000,000,002,002,002,006,032,000,000}, U(0279) }, /* Turned R */
 {{002,002,002,002,002,006,032,000,000}, U(027A) }, /* Turned long-leg R */
 {{000,000,002,002,002,006,032,002,001}, U(027B) }, /* Turned R, rigfht tail */
 {{000,000,013,014,010,010,010,010,010}, U(027C) }, /* Long-leg R */
 {{000,000,013,014,010,010,010,010,004}, U(027D) }, /* Right-tail R */
 {{000,000,006,011,010,010,010,000,000}, U(027E) }, /* Fish-hook R */
 {{000,000,036,021,036,022,021,000,000}, U(0280) }, /* Small capital R */
 {{000,000,021,022,036,021,036,000,000}, U(0281) }, /* Inverted small capital R */
 {{000,000,017,020,016,001,036,020,010}, U(0282) }, /* Right-tail S (at left) */
 {{002,004,004,004,004,004,004,004,010}, U(0283) }, /* Esh */
 {{002,004,004,004,004,016,004,004,010}, U(0284) }, /* Hooktop barred dotless J */
 {{002,004,004,004,004,016,024,024,010}, U(0286) }, /* Curly-tail esh */
 {{000,000,010,004,004,004,016,004,004}, U(0287) }, /* Turned T */
 {{004,004,016,004,004,004,004,004,002}, U(0288) }, /* Right-tail T */
 {{000,000,012,037,012,012,006,000,000}, U(0289) }, /* Barred U */
 {{000,000,033,012,021,021,016,000,000}, U(028A) }, /* Upsilon */
 {{000,000,032,011,011,012,014,000,000}, U(028B) }, /* Cursive V */
 {{000,000,004,012,012,021,021,000,000}, U(028C) }, /* Turned V */
 {{000,000,012,025,025,021,021,000,000}, U(028D) }, /* Turned W */
 {{016,020,034,022,021,021,021,000,000}, U(028E) }, /* Turned Y */
 {{000,000,021,012,004,004,004,000,000}, U(028F) }, /* Small capital Y */
 {{000,000,036,004,010,020,036,002,001}, U(0290) }, /* Right-tail Z */
 {{000,000,037,002,004,010,036,005,002}, U(0291) }, /* Curly-tail Z */
 {{000,000,037,002,004,016,001,021,016}, U(0292) }, /* Ezh; Tailed Z */
 {{000,000,037,002,004,002,011,025,016}, U(0293) }, /* Curly-tail ezh */
 {{016,021,020,020,020,020,020,021,016}, U(0297) }, /* Stretched C */
 {{000,000,036,021,036,021,036,000,000}, U(0299) }, /* Small capital B */
 {{016,021,002,004,004,004,004,000,000}, U(0294) }, /* Glottal stop */
 {{016,021,010,004,004,004,004,000,000}, U(0295) }, /* Reversed glottal stop */
 {{004,004,004,004,002,021,016,000,000}, U(0296) }, /* Inverted glottal stop */
 {{016,021,021,025,021,021,016,000,000}, U(0298) }, /* Bull's eye */
 {{000,000,016,021,015,021,016,000,000}, U(029A) }, /* Closed epsilon */
 {{001,002,016,020,026,022,016,000,000}, U(029B) }, /* Hooktop small capital G */
 {{000,000,021,021,037,021,021,000,000}, U(029C) }, /* Small capital H */
 {{004,000,004,004,004,016,024,024,010}, U(029D) }, /* Curly-tail J */
 {{000,000,022,012,006,012,022,002,002}, U(029E) }, /* Turned K */
 {{000,000,020,020,020,020,037,000,000}, U(029F) }, /* Small capital L */
 {{006,001,017,021,021,021,017,001,001}, U(02A0) }, /* Hooktop Q */
 {{016,021,002,004,037,004,004,000,000}, U(02A1) }, /* Barred glottal stop */
 {{016,021,010,004,037,004,004,000,000}, U(02A2) }, /* Barred reversed glottal stop */
 {{004,004,017,025,026,026,017,000,000}, U(02A3) }, /* dz ligature */
 {{004,004,017,025,026,025,015,001,006}, U(02A4) }, /* dezh ligature */
 {{004,004,017,025,026,026,017,005,002}, U(02A5) }, /* dz-curl ligature */
 {{020,020,037,024,022,021,016,000,000}, U(02A6) }, /* ts ligature */
 {{011,012,036,012,012,012,006,002,004}, U(02A7) }, /* tesh ligature */
 {{020,030,023,024,024,024,016,005,002}, U(02A8) }, /* tc-curl ligature */
 {{010,020,020,036,025,025,025,001,006}, U(02A9) }, /* feng ligature */
 {{030,010,013,014,012,011,036,000,000}, U(02AA) }, /* ls ligature */
 {{030,010,017,011,012,014,037,000,000}, U(02AB) }, /* lz ligature */
 {{000,000,025,012,000,025,012,000,000}, U(02AC) }, /* bilabial percussive */
 {{000,000,016,012,000,016,012,000,000}, U(02AD) }, /* bidental percussive */

 /* Spacing modifier letters */
 {{020,020,030,024,024,000,000,000,000}, U(02B0) }, /* Superscript H */
 {{010,000,010,010,010,020,000,000,000}, U(02B2) }, /* Superscript J */
 {{010,010,020,020,000,000,000,000,000}, U(02B9) }, /* modifier letter prime */
 {{011,011,022,022,000,000,000,000,000}, U(02BA) }, /* modifier double prime */
 {{004,004,010,000,000,000,000,000,000}, U(02BC) }, /* Modifier apostrophe */
 {{004,012,000,000,000,000,000,000,000}, 0x02c6, "circumflex" },
 {{012,004,000,000,000,000,000,000,000}, 0x02c7, "caron" },
 {{004,004,004,000,000,000,000,000,000}, U(02C8) }, /* Vertical stroke (superior) */
 {{000,000,000,000,000,000,004,004,004}, U(02CC) }, /* Vertical stroke (inferior) */
 {{000,000,012,004,000,004,012,000,000}, U(02D0) }, /* Length mark */
 {{000,000,012,004,000,000,000,000,000}, U(02D1) }, /* Half-length mark */
 {{021,016,000,000,000,000,000,000,000}, 0x02d8, "breve" },
 {{004,000,000,000,000,000,000,000,000}, 0x02d9, "dotaccent" },
 {{004,012,004,000,000,000,000,000,000}, 0x02da, "ring" },
 {{000,000,000,000,000,000,004,010,004}, 0x02db, "ogonek" },
 {{005,012,000,000,000,000,000,000,000}, 0x02dc, "tilde" },
 {{011,022,000,000,000,000,000,000,000}, 0x02dd, "hungarumlaut" },
 {{024,010,024,024,010,000,000,000,000}, U(02E0) }, /* Superscript gamma */
 {{030,010,010,010,034,000,000,000,000}, U(02E1) }, /* Superscript L */
 {{000,000,014,010,030,000,000,000,000}, U(02E2) }, /* Superscript S */
 {{000,000,024,010,024,000,000,000,000}, U(02E3) }, /* Superscript X */
 {{010,024,020,010,010,000,000,000,000}, U(02E4) }, /* Superscript reversed glottal stop */
 {{037,001,001,001,001,001,001,000,000}, U(02E5) }, /* Extra-high tone letter */
 {{001,037,001,001,001,001,001,000,000}, U(02E6) }, /* High tone letter */
 {{001,001,001,037,001,001,001,000,000}, U(02E7) }, /* Mid tone letter */
 {{001,001,001,001,001,037,001,000,000}, U(02E8) }, /* Low tone letter */
 {{001,001,001,001,001,001,037,000,000}, U(02E9) }, /* Extra-low tone letter */

 /* Greek and Coptic */
 {{001,001,001,001,001,021,016,000,000}, U(037F), }, /* Yot */
 {{004,000,000,000,000,000,000,000,000}, 0x0384, "tonos" },
 {{004,025,000,000,000,000,000,000,000}, 0x0385, "dieresistonos" },
 {{022,025,005,005,007,005,005,000,000}, 0x0386, "Alphatonos" },
 {{027,024,004,006,004,004,007,000,000}, 0x0388, "Epsilontonos" },
 {{025,025,005,007,005,005,005,000,000}, 0x0389, "Etatonos" },
 {{027,022,002,002,002,002,007,000,000}, 0x038a, "Iotatonos" },
 {{022,025,005,005,005,005,002,000,000}, 0x038c, "Omicrontonos" },
 {{025,025,005,002,002,002,002,000,000}, 0x038e, "Upsilontonos" },
 {{022,025,005,005,005,002,007,000,000}, 0x038f, "Omegatonos" },
 {{004,025,000,014,004,004,002,000,000}, 0x0390, "iotadieresistonos" },
 {{004,012,021,021,037,021,021,000,000}, 0x0391, "Alpha" },
 {{036,021,021,036,021,021,036,000,000}, 0x0392, "Beta" },
 {{037,020,020,020,020,020,020,000,000}, 0x0393, "Gamma" },
 {{004,004,012,012,021,021,037,000,000}, U(0394) }, /* Delta */
 {{037,020,020,036,020,020,037,000,000}, 0x0395, "Epsilon" },
 {{037,001,002,004,010,020,037,000,000}, 0x0396, "Zeta" },
 {{021,021,021,037,021,021,021,000,000}, 0x0397, "Eta" },
 {{016,021,021,037,021,021,016,000,000}, 0x0398, "Theta" },
 {{016,004,004,004,004,004,016,000,000}, 0x0399, "Iota" },
 {{021,022,024,030,024,022,021,000,000}, 0x039a, "Kappa" },
 {{004,004,012,012,021,021,021,000,000}, 0x039b, "Lambda" },
 {{021,033,025,025,021,021,021,000,000}, 0x039c, "Mu" },
 {{021,021,031,025,023,021,021,000,000}, 0x039d, "Nu" },
 {{037,000,000,016,000,000,037,000,000}, 0x039e, "Xi" },
 {{016,021,021,021,021,021,016,000,000}, 0x039f, "Omicron" },
 {{037,021,021,021,021,021,021,000,000}, 0x03a0, "Pi" },
 {{036,021,021,036,020,020,020,000,000}, 0x03a1, "Rho" },
 {{037,020,010,004,010,020,037,000,000}, 0x03a3, "Sigma" },
 {{037,004,004,004,004,004,004,000,000}, 0x03a4, "Tau" },
 {{021,021,012,004,004,004,004,000,000}, 0x03a5, "Upsilon" },
 {{004,016,025,025,025,016,004,000,000}, 0x03a6, "Phi" },
 {{021,021,012,004,012,021,021,000,000}, 0x03a7, "Chi" },
 {{025,025,025,025,025,016,004,000,000}, 0x03a8, "Psi" },
 {{016,021,021,021,021,012,033,000,000}, U(03A9) }, /* Omega */
 {{012,000,016,004,004,004,016,000,000}, 0x03aa, "Iotadieresis" },
 {{012,000,021,012,004,004,004,000,000}, 0x03ab, "Upsilondieresis" },
 {{004,000,015,022,022,022,015,000,000}, 0x03ac, "alphatonos" },
 {{004,000,017,020,016,020,017,000,000}, 0x03ad, "epsilontonos" },
 {{004,000,036,021,021,021,021,001,001}, 0x03ae, "etatonos" },
 {{004,000,014,004,004,004,002,000,000}, 0x03af, "iotatonos" },
 {{004,025,000,022,021,021,016,000,000}, 0x03b0, "upsilondieresistonos" },
 {{000,000,015,022,022,022,015,000,000}, 0x03b1, "alpha" },
 {{016,021,021,036,021,021,036,020,020}, 0x03b2, "beta" },
 {{000,000,021,021,012,012,004,004,004}, 0x03b3, "gamma" },
 {{017,020,016,021,021,021,016,000,000}, 0x03b4, "delta" },
 {{000,000,017,020,016,020,017,000,000}, 0x03b5, "epsilon" },
 {{036,002,004,010,020,020,014,002,004}, 0x03b6, "zeta" },
 {{000,000,036,021,021,021,021,001,001}, 0x03b7, "eta" },
 {{014,022,022,036,022,022,014,000,000}, 0x03b8, "theta" },
 {{000,000,014,004,004,004,002,000,000}, 0x03b9, "iota" },
 {{000,000,011,012,014,012,011,000,000}, 0x03ba, "kappa" },
 {{030,004,004,012,012,021,021,000,000}, 0x03bb, "lambda" },
 {{000,000,022,022,022,022,035,020,020}, U(03BC) }, /* mu */
 {{000,000,021,021,011,012,004,000,000}, 0x03bd, "nu" },
 {{016,020,020,016,020,020,016,001,002}, 0x03be, "xi" },
 {{000,000,016,021,021,021,016,000,000}, 0x03bf, "omicron" },
 {{000,000,037,012,012,012,011,000,000}, 0x03c0, "pi" },
 {{000,000,016,021,021,021,036,020,020}, 0x03c1, "rho" },
 {{000,000,017,020,020,020,016,001,002}, 0x03c2, "sigma1" },
 {{000,000,017,022,021,021,016,000,000}, 0x03c3, "sigma" },
 {{000,000,037,004,004,004,002,000,000}, 0x03c4, "tau" },
 {{000,000,022,021,021,021,016,000,000}, 0x03c5, "upsilon" },
 {{000,000,026,025,025,025,016,004,004}, 0x03c6, "phi" },
 {{000,000,021,011,012,004,012,022,021}, 0x03c7, "chi" },
 {{000,000,025,025,025,025,016,004,004}, 0x03c8, "psi" },
 {{000,000,012,021,025,025,012,000,000}, 0x03c9, "omega" },
 {{012,000,014,004,004,004,002,000,000}, 0x03ca, "iotadieresis" },
 {{012,000,022,021,021,021,016,000,000}, 0x03cb, "upsilondieresis" },
 {{004,000,016,021,021,021,016,000,000}, 0x03cc, "omicrontonos" },
 {{004,000,022,021,021,021,016,000,000}, 0x03cd, "upsilontonos" },
 {{004,000,012,021,025,025,012,000,000}, 0x03ce, "omegatonos" },
 {{004,004,016,025,025,025,016,004,004}, 0x03d5, "phi1" },
 {{004,000,004,004,004,004,004,004,010}, U(03F3) }, /* yot */

 /* Cyrillic */
 {{010,004,037,020,036,020,037,000,000}, U(0400) }, /* Ie grave */
 {{012,000,037,020,036,020,037,000,000}, U(0401) }, /* Yo */
 {{036,010,010,016,011,011,011,001,002}, U(0402) }, /* Dje */
 {{006,011,020,034,020,011,006,000,000}, U(0404) }, /* Ye Ukrainian */
 {{016,021,020,016,001,021,016,000,000}, U(0405) }, /* Dze */
 {{016,004,004,004,004,004,016,000,000}, U(0406) }, /* dotted I */
 {{012,000,016,004,004,004,016,000,000}, U(0407) }, /* Yi */
 {{001,001,001,001,001,021,016,000,000}, U(0408) }, /* Je */
 {{014,024,024,027,025,025,027,000,000}, U(0409) }, /* Lje */
 {{024,024,024,037,025,025,027,000,000}, U(040A) }, /* Nje */
 {{036,010,010,016,011,011,011,000,000}, U(040B) }, /* Tshe */
 {{025,021,021,037,001,001,037,000,000}, U(040E) }, /* short U */
 {{021,021,021,021,021,021,037,004,000}, U(040F) }, /* Dzhe */
 {{010,004,016,021,037,020,016,000,000}, U(0450) }, /* ie grave */
 {{012,000,016,021,037,020,016,000,000}, U(0451) }, /* yo */
 {{010,036,010,016,011,011,011,001,002}, U(0452) }, /* dje */
 {{000,000,014,022,030,022,014,000,000}, U(0454) }, /* ye Ukrainian */
 {{000,000,017,020,016,001,036,000,000}, U(0455) }, /* dze */
 {{004,000,014,004,004,004,016,000,000}, U(0456) }, /* dotted i */
 {{012,000,014,004,004,004,016,000,000}, U(0457) }, /* yi */
 {{004,000,004,004,004,004,004,004,010}, U(0458) }, /* je */
 {{000,000,014,024,026,025,026,000,000}, U(0459) }, /* lje */
 {{000,000,024,024,036,025,026,000,000}, U(045A) }, /* nje */
 {{010,036,010,016,011,011,011,000,000}, U(045B) }, /* tshe */
 {{000,004,021,021,021,021,017,001,016}, U(045E) }, /* short u */
 {{000,000,021,021,021,021,037,004,000}, U(045F) }, /* dzhe */
 {{001,037,020,020,020,020,020,000,000}, U(0490) }, /* Ghe with upturn */
 {{000,001,037,020,020,020,020,000,000}, U(0491) }, /* ghe with upturn */
 {{017,010,010,036,010,010,010,000,000}, U(0492) }, /* Ghe with stroke */
 {{000,000,017,010,036,010,010,000,000}, U(0493) }, /* ghe with stroke */
 
#ifdef ARMENIAN
 /* Armenian */
 {{021,021,021,021,025,022,015,000,000}, U(0531) }, /* Ayb */
 {{016,021,021,020,037,020,020,000,000}, U(0532) }, /* Ben */
 {{014,022,022,022,017,002,002,000,000}, U(0533) }, /* Gim */
 {{014,022,022,002,003,002,002,000,000}, U(0534) }, /* Da */
 {{020,020,037,020,021,021,016,000,000}, U(0535) }, /* Ech */
 {{016,021,021,001,001,022,037,000,000}, U(0536) }, /* Za */
 {{020,020,020,037,020,020,036,000,000}, U(0537) }, /* Eh */
 {{016,021,021,020,020,020,037,000,000}, U(0538) }, /* Et */
 {{016,021,021,023,025,025,022,000,000}, U(0539) }, /* To */
 {{002,002,017,022,022,022,014,000,000}, U(053A) }, /* Zhe */
 {{020,020,036,021,021,021,020,000,000}, U(053B) }, /* Ini */
 {{020,020,020,020,020,020,037,000,000}, U(053C) }, /* Liwn */
 {{020,020,035,025,025,025,022,000,000}, U(053D) }, /* Xeh */
 {{037,016,021,021,021,021,016,000,000}, U(053E) }, /* Ca */
 {{020,021,021,017,001,001,001,000,000}, U(053F) }, /* Ken */
 {{002,001,003,014,020,014,003,000,000}, U(0540) }, /* Ho */
 {{016,021,021,001,015,022,015,000,000}, U(0541) }, /* Ja */
 {{014,022,022,002,002,002,003,000,000}, U(0542) }, /* Ghad */
 {{001,032,004,012,012,021,037,000,000}, U(0543) }, /* Cheh */
 {{023,022,022,022,022,022,014,000,000}, U(0544) }, /* Men */
 {{016,021,001,036,001,021,016,000,000}, U(0545) }, /* Yi */
 {{030,010,010,010,011,011,006,000,000}, U(0546) }, /* Now */
 {{030,007,010,020,021,021,016,000,000}, U(0547) }, /* Sha */
 {{016,021,021,021,021,021,021,000,000}, U(0548) }, /* Vo */
 {{016,021,021,001,002,034,003,000,000}, U(0549) }, /* Cha */
 {{016,025,025,005,005,001,001,000,000}, U(054A) }, /* Peh */
 {{016,021,021,031,005,026,037,000,000}, U(054B) }, /* Jheh */
 {{014,022,022,023,022,022,022,000,000}, U(054C) }, /* Ra */
 {{021,021,021,021,021,021,016,000,000}, U(054D) }, /* Seh */
 {{002,022,022,022,016,002,003,000,000}, U(054E) }, /* Vew */
 {{016,021,020,016,001,021,016,000,000}, U(054F) }, /* Tiwn */
 {{016,021,021,020,020,020,020,000,000}, U(0550) }, /* Reh */
 {{016,021,016,021,001,021,016,000,000}, U(0551) }, /* Co */
 {{020,020,037,020,020,020,020,000,000}, U(0552) }, /* Yiwn */
 {{004,016,025,025,025,016,004,000,000}, U(0553) }, /* Piwr */
 {{006,011,011,016,010,037,010,000,000}, U(0554) }, /* Keh */
 {{016,021,021,021,021,021,016,000,000}, U(0555) }, /* Oh */
 {{014,024,016,005,025,025,016,000,000}, U(0556) }, /* Feh */
 {{000,000,025,025,025,025,013,000,000}, U(0561) }, /* ayb */
 {{000,000,036,021,021,020,037,020,020}, U(0562) }, /* ben */
 {{000,000,016,022,022,022,017,002,002}, U(0563) }, /* gim */
 {{000,000,034,022,022,022,023,002,002}, U(0564) }, /* da */
 {{020,020,037,020,021,021,017,000,000}, U(0565) }, /* ech */
 {{000,000,016,022,022,022,016,002,003}, U(0566) }, /* za */
 {{020,020,037,020,020,020,036,000,000}, U(0567) }, /* eh */
 {{000,000,036,021,021,021,021,020,037}, U(0568) }, /* et */
 {{000,000,036,021,027,031,026,020,020}, U(0569) }, /* to */
 {{002,002,017,022,022,022,014,000,000}, U(056A) }, /* zhe */
 {{020,020,036,021,021,021,021,020,020}, U(056B) }, /* ini */
 {{000,000,010,010,010,010,010,010,016}, U(056C) }, /* liwn */
 {{020,020,035,025,025,025,023,020,020}, U(056D) }, /* xeh */
 {{004,002,017,022,022,022,014,000,000}, U(056E) }, /* ca */
 {{020,020,021,021,021,021,017,001,001}, U(056F) }, /* ken */
 {{020,020,036,021,021,021,021,000,000}, U(0570) }, /* ho */
 {{004,002,004,012,021,023,015,000,000}, U(0571) }, /* ja */
 {{000,000,034,022,022,022,022,002,003}, U(0572) }, /* ghad */
 {{007,010,036,011,011,011,007,000,000}, U(0573) }, /* cheh */
 {{003,002,022,022,022,022,016,000,000}, U(0574) }, /* men */
 {{000,000,004,004,004,004,004,004,010}, U(0575) }, /* yi */
 {{030,010,011,011,011,011,007,000,000}, U(0576) }, /* now */
 {{000,000,016,021,001,002,014,020,037}, U(0577) }, /* sha */
 {{000,000,036,021,021,021,021,000,000}, U(0578) }, /* vo */
 {{000,000,004,010,004,002,014,020,037}, U(0579) }, /* cha */
 {{000,000,025,025,025,025,013,001,001}, U(057A) }, /* peh */
 {{000,000,016,021,021,012,014,020,037}, U(057B) }, /* jheh */
 {{000,000,036,021,021,022,023,000,000}, U(057C) }, /* ra */
 {{000,000,021,021,021,021,017,000,000}, U(057D) }, /* seh */
 {{002,002,022,022,022,022,016,002,003}, U(057E) }, /* vew */
 {{000,000,026,025,025,025,015,000,000}, U(057F) }, /* tiwn */
 {{000,000,036,021,021,021,021,020,020}, U(0580) }, /* reh */
 {{000,000,017,021,021,021,017,001,016}, U(0581) }, /* co */
 {{000,000,010,010,010,010,016,000,000}, U(0582) }, /* yiwn */
 {{004,004,026,025,025,025,015,004,004}, U(0583) }, /* piwr */
 {{000,000,016,011,011,016,010,037,010}, U(0584) }, /* keh */
 {{000,000,016,021,021,021,016,000,000}, U(0585) }, /* oh */
 {{014,024,016,005,025,025,016,004,004}, U(0586) }, /* feh */
 {{020,020,024,024,024,024,017,000,000}, U(0587) }, /* ech_yiwn */
 {{014,022,022,007,002,007,002,000,000}, U(058F) }, /* armdram */
#endif
 
 /* Phonetic extensions */
 {{000,000,016,021,037,021,021,000,000}, U(1D00) }, /* small cap A */
 {{000,000,017,024,026,034,027,000,000}, U(1D01) }, /* small cap AE */
 {{000,000,016,005,036,024,012,000,000}, U(1D02) }, /* turned ae */
 {{000,000,016,021,020,021,016,000,000}, U(1D04) }, /* small cap C */
 {{000,000,036,021,021,021,036,000,000}, U(1D05) }, /* small cap D */
 {{000,000,016,011,035,011,016,000,000}, U(1D06) }, /* small cap Eth */
 {{000,000,037,020,036,020,037,000,000}, U(1D07) }, /* small cap E */
 {{000,000,016,004,004,004,006,000,004}, U(1D09) }, /* turned i */
 {{000,000,001,001,001,021,016,000,000}, U(1D0A) }, /* small cap J */
 {{000,000,021,022,034,022,021,000,000}, U(1D0B) }, /* small cap K */
 {{000,000,021,033,025,021,021,000,000}, U(1D0D) }, /* small cap M */
 {{000,000,021,023,025,031,021,000,000}, U(1D0E) }, /* small cap reversed N */
 {{000,000,016,021,021,021,016,000,000}, U(1D0F) }, /* small cap O */
 {{000,000,032,005,035,025,012,000,000}, U(1D14) }, /* turned oe */
 {{000,000,036,021,036,020,020,000,000}, U(1D18) }, /* small cap P */
 {{000,000,017,021,017,011,021,000,000}, U(1D19) }, /* small cap reversed R */
 {{000,000,021,011,017,021,017,000,000}, U(1D1A) }, /* small cap turned R */
 {{000,000,037,004,004,004,004,000,000}, U(1D1B) }, /* small cap T */
 {{000,000,021,021,021,021,016,000,000}, U(1D1C) }, /* small cap U */
 {{000,000,037,001,001,001,036,000,000}, U(1D1D) }, /* sideways u */
 {{000,000,037,001,036,001,036,000,000}, U(1D1F) }, /* sideways m */
 {{000,000,021,021,012,012,004,000,000}, U(1D20) }, /* small cap V */
 {{000,000,021,021,025,025,012,000,000}, U(1D21) }, /* small cap W */
 {{000,000,037,001,016,020,037,000,000}, U(1D22) }, /* small cap Z */
 {{030,004,034,024,010,000,000,000,000}, U(1D4A) }, /* Superscript schwa */
 {{000,000,000,000,004,000,014,004,016}, U(1D62) }, /* iinferior */

 /* Phonetic extensions supplement */
 {{010,024,034,024,010,000,000,000,000}, U(1DBF) }, /* Superscript theta */

 /* Latin extended additional */
 {{036,021,021,036,021,021,036,000,004}, U(1E04) }, /* Bdotbelow */
 {{020,020,036,021,021,021,036,000,004}, U(1E05) }, /* bdotbelow */
 {{036,021,021,036,021,021,036,000,016}, U(1E06) }, /* Bmacronbelow */
 {{020,020,036,021,021,021,036,000,016}, U(1E07) }, /* bmacronbelow */
 {{036,021,021,021,021,021,036,000,004}, U(1E0C) }, /* Ddotbelow */
 {{001,001,017,021,021,021,017,000,004}, U(1E0D) }, /* ddotbelow */
 {{036,021,021,021,021,021,036,000,016}, U(1E0E) }, /* Dmacronbelow */
 {{001,001,017,021,021,021,017,000,016}, U(1E0F) }, /* dmacronbelow */
 {{016,000,017,020,023,021,017,000,000}, U(1E20) }, /* Gmacron */
 {{016,000,017,021,021,021,017,001,016}, U(1E21) }, /* gmacron */
 {{021,021,021,037,021,021,021,000,004}, U(1E24) }, /* Hdotbelow */
 {{020,020,036,021,021,021,021,000,004}, U(1E25) }, /* hdotbelow */
 {{012,000,021,021,037,021,021,000,000}, U(1E26) }, /* Hdieresis */
 {{025,020,036,021,021,021,021,000,000}, U(1E27) }, /* hdieresis */
 {{021,022,024,030,024,022,021,000,004}, U(1E32) }, /* Kdotbelow */
 {{010,010,011,012,014,012,011,000,004}, U(1E33) }, /* kdotbelow */
 {{021,022,024,030,024,022,021,000,016}, U(1E34) }, /* Kmacronbelow */
 {{010,010,011,012,014,012,011,000,016}, U(1E35) }, /* kmacronbelow */
 {{020,020,020,020,020,020,037,000,004}, U(1E36) }, /* Ldotbelow */
 {{014,004,004,004,004,004,016,000,004}, U(1E37) }, /* ldotbelow */
 {{020,020,020,020,020,020,037,000,016}, U(1E3A) }, /* Lmacronbelow */
 {{014,004,004,004,004,004,016,000,016}, U(1E3B) }, /* lmacronbelow */
 {{021,033,025,025,021,021,021,000,004}, U(1E42) }, /* Mdotbelow */
 {{000,000,032,025,025,025,025,000,004}, U(1E43) }, /* mdotbelow */
 {{021,021,031,025,023,021,021,000,004}, U(1E46) }, /* Ndotbelow */
 {{000,000,036,021,021,021,021,000,004}, U(1E47) }, /* ndotbelow */
 {{021,021,031,025,023,021,021,000,016}, U(1E48) }, /* Nmacronbelow */
 {{000,000,036,021,021,021,021,000,016}, U(1E49) }, /* nmacronbelow */
 {{036,021,021,036,024,022,021,000,004}, U(1E5A) }, /* Rdotbelow */
 {{000,000,013,014,010,010,010,000,004}, U(1E5B) }, /* rdotbelow */
 {{016,000,036,021,036,022,021,000,004}, U(1E5C) }, /* Rdotbelowmacron */
 {{016,000,013,014,010,010,010,000,004}, U(1E5D) }, /* rdotbelowmacron */
 {{036,021,021,036,024,022,021,000,016}, U(1E5E) }, /* Rmacronbelow */
 {{000,000,013,014,010,010,010,000,016}, U(1E5F) }, /* rmacronbelow */
 {{016,021,020,016,001,021,016,000,004}, U(1E62) }, /* Sdotbelow */
 {{000,000,017,020,016,001,036,000,004}, U(1E63) }, /* sdotbelow */
 {{037,004,004,004,004,004,004,000,004}, U(1E6C) }, /* Tdotbelow */
 {{004,004,016,004,004,004,002,000,004}, U(1E6D) }, /* tdotbelow */
 {{037,004,004,004,004,004,004,000,016}, U(1E6E) }, /* Tmacronbelow */
 {{004,004,016,004,004,004,002,000,016}, U(1E6F) }, /* tmacronbelow */
 {{021,021,021,021,021,021,016,000,012}, U(1E72) }, /* Udieresisbelow */
 {{000,000,021,021,021,021,017,000,012}, U(1E73) }, /* udieresisbelow */
 {{021,021,021,012,012,004,004,000,004}, U(1E7E) }, /* Vdotbelow */
 {{000,000,021,021,012,012,004,000,004}, U(1E7F) }, /* vdotbelow */
 {{012,000,021,025,025,025,012,000,000}, 0x1e84, "Wdieresis", },
 {{012,000,021,021,025,025,012,000,000}, 0x1e85, "wdieresis", },
 {{021,021,021,025,025,025,012,000,004}, U(1E88) }, /* Wdotbelow */
 {{000,000,021,021,025,025,012,000,004}, U(1E89) }, /* wdotbelow */
 {{012,000,021,021,016,021,021,000,000}, U(1E8C) }, /* Xdieresis */
 {{012,000,021,012,004,012,021,000,000}, U(1E8D) }, /* xdieresis */
 {{037,001,002,004,010,020,037,000,004}, U(1E92) }, /* Zdotbelow */
 {{000,000,037,002,004,010,037,000,004}, U(1E93) }, /* zdotbelow */
 {{037,001,002,004,010,020,037,000,016}, U(1E94) }, /* Zmacronbelow */
 {{000,000,037,002,004,010,037,000,016}, U(1E95) }, /* zmacronbelow */
 {{020,020,036,021,021,021,021,000,016}, U(1E96) }, /* Hmacronbelow */
 {{025,004,016,004,004,004,002,000,000}, U(1E97) }, /* tdieresis */
 {{016,021,022,022,021,021,026,000,000}, U(1E9E) }, /* Germandbls */
 {{004,012,021,021,037,021,021,000,004}, U(1EA0) }, /* Adotbelow */
 {{000,000,016,001,017,021,017,000,004}, U(1EA1) }, /* adotbelow */
 {{037,020,020,036,020,020,037,000,004}, U(1EB8) }, /* Edotbelow */
 {{000,000,016,021,037,020,016,000,004}, U(1EB9) }, /* edotbelow */
 {{016,004,004,004,004,004,016,000,004}, U(1ECA) }, /* Idotbelow */
 {{004,000,014,004,004,004,016,000,004}, U(1ECB) }, /* idotbelow */
 {{016,021,021,021,021,021,016,000,004}, U(1ECC) }, /* Odotbelow */
 {{000,000,016,021,021,021,016,000,004}, U(1ECD) }, /* odotbelow */
 {{021,021,021,021,021,021,016,000,004}, U(1EE4) }, /* Udotbelow */
 {{000,000,021,021,021,021,017,000,004}, U(1EE5) }, /* udotbelow */
 {{021,021,012,004,004,004,004,000,004}, U(1EF4) }, /* Ydotbelow */
 {{000,000,021,021,017,001,016,000,004}, U(1EF5) }, /* ydotbelow */
 
 /* General punctuation */
 {{000,000,000,016,000,000,000,000,000}, U(2010) }, /* hyphen */
 {{000,000,000,016,000,000,000,000,000}, U(2011) }, /* non-breaking hyphen */
 {{000,000,000,036,000,000,000,000,000}, 0x2013, "endash" },
 {{000,000,000,000,000,000,037,000,037}, 0x2017, "underscoredbl" },
 {{002,004,004,000,000,000,000,000,000}, 0x2018, "quoteleft" },
 {{000,000,000,000,000,004,004,010,000}, 0x201a, "quotesinglbase" },
 {{011,022,022,000,000,000,000,000,000}, 0x201c, "quotedblleft" },
 {{011,011,022,000,000,000,000,000,000}, 0x201d, "quotedblright" },
 {{000,000,000,000,000,011,011,022,000}, 0x201e, "quotedblbase" },
 {{022,022,011,000,000,000,000,000,000}, U(201F) }, /* quotedblreversed */
 {{004,004,037,004,004,004,004,004,004}, 0x2020, "dagger" },
 {{004,004,037,004,004,004,037,004,004}, 0x2021, "daggerdbl" },
 {{000,000,016,016,016,000,000,000,000}, 0x2022, "bullet" },
 {{000,000,000,000,000,000,025,000,000}, 0x2026, "ellipsis" },
 {{021,022,004,010,020,005,005,000,000}, 0x2030, "perthousand" },
 {{010,010,020,020,000,000,000,000,000}, 0x2032, "minute" },
 {{011,011,022,022,000,000,000,000,000}, 0x2033, "second" },
 {{002,002,001,001,000,000,000,000,000}, U(2035) }, /* reversed prime */
 {{022,022,011,011,000,000,000,000,000}, U(2036) }, /* reversed double prime */
 {{000,000,000,002,004,002,000,000,000}, 0x2039, "guilsinglleft" },
 {{000,000,000,010,004,010,000,000,000}, 0x203a, "guilsinglright" },
 {{025,021,012,004,025,004,012,021,025}, U(203B) }, /* referencemark */
 {{012,012,012,012,012,000,012,000,000}, 0x203c, "exclamdbl" },
 {{016,025,005,006,004,000,004,000,000}, U(203D) }, /* interrobang */
 {{037,000,000,000,000,000,000,000,000}, U(203E) }, /* overline */
 {{000,000,000,000,000,000,000,021,016}, U(203F) }, /* Bottom tie bar */
 {{016,021,000,000,000,000,000,000,000}, U(2040) }, /* character tie */
 {{001,001,002,002,004,010,010,020,020}, 0x2044, "fraction" },
 {{012,025,005,011,012,000,012,000,000}, U(2047) }, /* questiondbl */
 {{011,025,005,011,011,000,011,000,000}, U(2048) }, /* question exclam */
 {{022,025,021,022,022,000,022,000,000}, U(2049) }, /* exclam question */
 {{026,025,025,026,024,024,024,000,000}, U(204B) }, /* reversed pilcrow */
 {{000,000,004,000,000,004,004,002,000}, U(204F) }, /* reversed semicolon */
 {{021,001,002,004,010,020,021,000,000}, U(2052) }, /* commercial minus */
 {{000,000,010,025,002,000,000,000,000}, U(2053) }, /* swing dash */
 {{000,000,000,000,000,000,000,016,021}, U(2054) }, /* inverted undertie */
 {{000,000,001,000,020,000,001,000,000}, U(2056) }, /* three dots */
 {{000,000,004,000,021,000,004,000,000}, U(2058) }, /* four dots */
 {{000,000,021,000,004,000,021,000,000}, U(2059) }, /* five dots */
 {{004,000,000,000,000,000,004,000,000}, U(205A) }, /* two dots */
 {{004,000,000,000,021,000,000,000,004}, U(205B) }, /* large four dots */
 {{000,000,025,004,037,004,025,000,000}, U(205C) }, /* dotted cross */
 {{004,000,000,004,000,000,004,000,000}, U(205D) }, /* tricolon */
 {{004,000,004,000,004,000,004,000,000}, U(205E) }, /* vertical four dots */

 /* Subscripts and superscripts */
 {{004,012,012,012,004,000,000,000,000}, U(2070) }, /* zerosuperior */
 {{004,000,014,004,016,000,000,000,000}, U(2071) }, /* isuperior */
 {{002,006,012,016,002,000,000,000,000}, U(2074) }, /* foursuperior */
 {{016,010,014,002,014,000,000,000,000}, U(2075) }, /* fivesuperior */
 {{004,010,014,012,004,000,000,000,000}, U(2076) }, /* sixsuperior */
 {{016,002,004,010,010,000,000,000,000}, U(2077) }, /* sevensuperior */
 {{004,012,004,012,004,000,000,000,000}, U(2078) }, /* eightsuperior */
 {{004,012,006,002,004,000,000,000,000}, U(2079) }, /* ninesuperior */
 {{000,004,016,004,000,000,000,000,000}, U(207A) }, /* plussuperior */
 {{000,000,016,000,000,000,000,000,000}, U(207B) }, /* minussuperior */
 {{000,016,000,016,000,000,000,000,000}, U(207C) }, /* equalssuperior */
 {{002,004,004,004,002,000,000,000,000}, U(207D) }, /* parenleftsuperior */
 {{010,004,004,004,010,000,000,000,000}, U(207E) }, /* parenrightsuperior */
 {{000,000,014,012,012,000,000,000,000}, U(207F) }, /* nsuperior */
 {{000,000,000,000,004,012,012,012,004}, U(2080) }, /* zeroinferior */
 {{000,000,000,000,004,014,004,004,016}, U(2081) }, /* oneinferior */
 {{000,000,000,000,014,002,004,010,016}, U(2082) }, /* twoinferior */
 {{000,000,000,000,014,002,014,002,014}, U(2083) }, /* threeinferior */
 {{000,000,000,000,002,006,012,016,002}, U(2084) }, /* fourinferior */
 {{000,000,000,000,016,010,014,002,014}, U(2085) }, /* fiveinferior */
 {{000,000,000,000,004,010,014,012,004}, U(2086) }, /* sixinferior */
 {{000,000,000,000,016,002,004,010,010}, U(2087) }, /* seveninferior */
 {{000,000,000,000,004,012,004,012,004}, U(2088) }, /* eightinferior */
 {{000,000,000,000,004,012,006,002,004}, U(2089) }, /* nineinferior */
 {{000,000,000,000,000,004,016,004,000}, U(208A) }, /* plusinferior */
 {{000,000,000,000,000,000,016,000,000}, U(208B) }, /* minusinferior */
 {{000,000,000,000,000,016,000,016,000}, U(208C) }, /* equalsinferior */
 {{000,000,000,000,002,004,004,004,002}, U(208D) }, /* parenleftinferior */
 {{000,000,000,000,010,004,004,004,010}, U(208E) }, /* parenrightinferior */
 {{000,000,000,000,000,000,014,012,012}, U(2099) }, /* ninferior */

 /* Currency symbols */
 {{030,024,030,022,027,022,021,000,000}, 0x20a7, "peseta" },
 {{031,025,025,021,025,025,026,000,000}, U(20AA) }, /* newsheqel */
 {{006,011,034,010,034,011,006,000,000}, 0x20ac, "Euro" },

 /* Letterlike symbols */
 {{016,021,035,033,033,035,031,021,016}, U(2117) }, /* phonographic */
 {{016,021,021,021,021,012,033,000,000}, 0x2126, "Omega" },
 {{033,012,021,021,021,021,016,000,000}, U(2127) }, /* mho sign */
 {{000,000,010,004,004,004,006,000,000}, U(2129) }, /* turned iota */
 {{001,001,001,017,001,001,037,000,000}, U(2132) }, /* turned F */
 {{026,011,025,002,005,005,002,000,000}, U(214B) }, /* turned ampersand */
 {{000,000,001,001,017,001,037,000,000}, U(214E) }, /* small cap turned F */

 /* Number forms */
 {{020,020,020,020,027,001,002,004,004}, U(2150) }, /* oneseventh */
 {{020,020,020,020,022,005,003,001,002}, U(2151) }, /* oneninth */
 {{010,010,010,010,000,022,025,025,022}, U(2152) }, /* onetenth */
 {{020,020,020,020,026,001,006,001,006}, 0x2153, "onethird" },
 {{020,010,010,020,032,001,002,001,002}, 0x2154, "twothirds" },
 {{020,020,020,020,027,004,006,001,006}, U(2155) }, /* onefifth */
 {{020,010,010,020,033,002,003,001,002}, U(2156) }, /* twofifths */
 {{020,010,020,010,023,002,003,001,002}, U(2157) }, /* threefifths */
 {{004,014,024,034,007,002,003,001,002}, U(2158) }, /* fourfifths */
 {{020,020,020,020,022,004,006,005,002}, U(2159) }, /* onesixth */
 {{030,020,030,010,022,004,006,005,002}, U(215A) }, /* fivesixths */
 {{020,020,020,020,022,005,002,005,002}, 0x215b, "oneeighth" },
 {{020,010,020,010,022,005,002,005,002}, 0x215c, "threeeighths" },
 {{030,020,030,010,022,005,002,005,002}, 0x215d, "fiveeighths" },
 {{034,004,010,020,022,005,002,005,002}, 0x215e, "seveneighths" },
 {{010,024,024,010,002,001,002,001,002}, U(2189) }, /* zerothirds */
 {{037,001,002,014,020,021,016,000,000}, U(218A) }, /* turned 2 */
 {{016,021,020,014,010,020,037,000,000}, U(218B) }, /* turned 3 */

 /* Arrows */
 {{000,004,004,025,016,004,000,000,000}, 0x2193, "arrowdown" },
 {{000,012,021,037,021,012,000,000,000}, 0x2194, "arrowboth" },
 {{004,016,025,004,025,016,004,000,000}, 0x2195, "arrowupdn" },
 {{000,036,030,024,022,001,000,000,000}, U(2196) }, /* arrowupleft */
 {{000,017,003,005,011,020,000,000,000}, U(2197) }, /* arrowupright */
 {{000,020,011,005,003,017,000,000,000}, U(2198) }, /* arrowdownright */
 {{000,001,022,024,030,036,000,000,000}, U(2199) }, /* arrowdownleft */
 {{000,005,011,037,011,005,000,000,000}, U(21A4) }, /* arrowbarleft */
 {{000,004,016,025,004,037,000,000,000}, U(21A5) }, /* arrowbarup" */
 {{000,024,022,037,022,024,000,000,000}, U(21A6) }, /* arrowbarright */
 {{000,037,004,025,016,004,000,000,000}, U(21A7) }, /* arrowbardown */
 {{004,016,025,004,025,016,037,000,000}, 0x21a8, "arrowupdnbse" },
 {{000,034,004,025,016,004,000,000,000}, U(21B4) }, /* arrow right then down */
 {{001,005,011,037,010,004,000,000,000}, 0x21b5, "carriagereturn" },
 {{024,030,037,030,025,003,037,003,005}, U(21B9) }, /* arrowtabsleftright */
 {{000,004,010,037,000,000,000,000,000}, U(21BC) }, /* harpoonleftbarbup */
 {{000,000,000,037,010,004,000,000,000}, U(21BD) }, /* harpoonleftbarbdown */
 {{000,004,006,005,004,004,000,000,000}, U(21BE) }, /* harpoonupbarbright */
 {{000,004,014,024,004,004,000,000,000}, U(21BF) }, /* harpoonupbarbleft */
 {{000,004,002,037,000,000,000,000,000}, U(21C0) }, /* harpoonrightbarbup */
 {{000,000,000,037,002,004,000,000,000}, U(21C1) }, /* harpoonrightbarbdown */
 {{000,004,004,005,006,004,000,000,000}, U(21C2) }, /* harpoondownbarbright */
 {{000,004,004,024,014,004,000,000,000}, U(21C3) }, /* harpoondownbarbleft */
 {{004,010,037,000,037,002,004,000,000}, U(21CB) }, /* harpoonsleftright */
 {{004,002,037,000,037,010,004,000,000}, U(21CC) }, /* harpoonsrightleft */
 {{000,006,013,020,013,006,000,000,000}, 0x21d0, "arrowdblleft" },
 {{004,012,021,033,012,012,012,000,000}, 0x21d1, "arrowdblup" },
 {{000,014,032,001,032,014,000,000,000}, 0x21d2, "arrowdblright" },
 {{012,012,012,033,021,012,004,000,000}, 0x21d3, "arrowdbldown" },
 {{000,024,030,037,030,024,000,000,000}, U(21E4) }, /* arrowtableft */
 {{000,005,003,037,003,005,000,000,000}, U(21E5) }, /* arrowtabright */
 {{000,006,013,021,013,006,000,000,000}, U(21E6) }, /* white left arrow */
 {{004,012,021,033,012,012,016,000,000}, U(21E7) }, /* white up arrow */
 {{000,014,032,021,032,014,000,000,000}, U(21E8) }, /* white right arrow */
 {{016,012,012,033,021,012,004,000,000}, U(21E9) }, /* white down arrow */
 {{004,012,021,033,016,012,016,000,000}, U(21EA) }, /* caps lock arrow */

 /* Mathematical operators */
 {{016,021,001,017,021,021,016,000,000}, 0x2202, "partialdiff" },
 {{004,004,012,012,021,021,037,000,000}, 0x2206, "Delta" },
 {{037,021,021,012,012,004,004,000,000}, 0x2207, "gradient" },
 {{017,020,020,037,020,020,017,000,000}, 0x2208, "element" },
 {{000,017,020,037,020,017,000,000,000}, U(220A) }, /* small element of */
 {{036,001,001,037,001,001,036,000,000}, 0x220b, "suchthat" },
 {{000,036,001,037,001,036,000,000,000}, U(220D) }, /* small has element */
 {{037,021,021,021,021,021,021,021,021}, 0x220f, "product" },
 {{021,021,021,021,021,021,021,021,037}, U(2210) }, /* n-ary coproduct */
 {{037,020,010,004,002,004,010,020,037}, 0x2211, "summation" },
 {{000,000,000,037,000,000,000,000,000}, 0x2212, "minus" },
 {{037,000,004,004,037,004,004,000,000}, U(2213) }, /* minusplus */
 {{004,000,004,004,037,004,004,000,000}, U(2214) }, /* dot plus */
 {{000,000,004,012,012,004,000,000,000}, U(2218) }, /* ring operator */
 {{000,000,004,016,004,000,000,000,000}, U(2219) }, /* bulletoperator */
 {{007,004,004,004,024,014,004,000,000}, 0x221a, "radical" },
 {{023,012,022,012,022,006,002,000,000}, U(221B) }, /* cube root */
 {{015,025,035,005,001,003,001,000,000}, U(221C) }, /* fourth root */
 {{000,012,025,024,025,012,000,000,000}, 0x221d, "proportional" },
 {{000,012,025,025,025,012,000,000,000}, 0x221e, "infinity" },
 {{000,020,020,020,020,037,000,000,000}, 0x221f, "orthogonal" },
 {{004,004,004,004,004,004,004,000,000}, U(2223) }, /* divides */
 {{012,012,012,012,012,012,012,000,000}, U(2225) }, /* parallel */
 {{000,000,004,012,021,000,000,000,000}, 0x2227, "logicaland" },
 {{000,000,021,012,004,000,000,000,000}, 0x2228, "logicalor" },
 {{000,016,021,021,021,021,000,000,000}, 0x2229, "intersection" },
 {{000,021,021,021,021,016,000,000,000}, 0x222a, "union" },
 {{002,005,004,004,004,004,004,024,010}, 0x222b, "integral" },
 {{002,005,004,016,025,016,004,024,010}, U(222E) }, /* contour integral */
 {{000,004,000,000,000,021,000,000,000}, 0x2234, "therefore" },
 {{000,021,000,000,000,004,000,000,000}, U(2235) }, /* because */
 {{000,004,000,037,000,000,000,000,000}, U(2238) }, /* dot minus */
 {{000,021,000,037,000,021,000,000,000}, U(223A) }, /* geometric proportion */
 {{000,000,010,025,002,000,000,000,000}, 0x223c, "similar" },
 {{000,000,002,025,010,000,000,000,000}, U(223D) }, /* reversed tilde */
 {{000,010,024,025,005,002,000,000,000}, U(223F) }, /* sinewave */
 {{000,037,000,010,025,002,000,000,000}, U(2242) }, /* minus tilde */
 {{000,010,025,002,000,037,000,000,000}, U(2243) }, /* asymptotically equal */
 {{010,025,002,000,037,000,037,000,000}, 0x2245, "congruent" },
 {{000,010,025,002,010,025,002,000,000}, 0x2248, "approxequal" },
 {{002,025,010,000,037,000,037,000,000}, U(224C) }, /* all equal to */
 {{000,021,016,000,016,021,000,000,000}, U(224D) }, /* equivalent to */
 {{000,004,033,000,033,004,000,000,000}, U(224E) }, /* geom equiv to */
 {{000,004,033,000,037,000,000,000,000}, U(224F) }, /* difference between */
 {{004,000,037,000,037,000,000,000,000}, U(2250) }, /* approaches limit */
 {{004,000,037,000,037,000,004,000,000}, U(2251) }, /* geometrically equal */
 {{020,000,037,000,037,000,001,000,000}, U(2252) }, /* approx eq or image */
 {{001,000,037,000,037,000,020,000,000}, U(2253) }, /* image or approx eq */
 {{016,021,000,037,000,037,000,000,000}, U(2258) }, /* corresponds to */
 {{032,025,025,000,037,000,037,000,000}, U(225E) }, /* measured by */
 {{000,002,037,004,037,010,000,000,000}, 0x2260, "notequal" },
 {{000,037,000,037,000,037,000,000,000}, 0x2261, "equivalence" },
 {{002,037,004,037,004,037,010,000,000}, U(2262) }, /* not identical */
 {{003,014,020,014,003,030,007,000,000}, 0x2264, "lessequal" },
 {{030,006,001,006,030,003,034,000,000}, 0x2265, "greaterequal" },
 {{012,004,012,012,012,004,012,000,000}, U(226C) }, /* between */
 {{000,017,020,020,020,017,000,000,000}, 0x2282, "propersubset" },
 {{000,036,001,001,001,036,000,000,000}, 0x2283, "propersuperset" },
 {{017,020,020,020,017,000,037,000,000}, 0x2286, "reflexsubset" },
 {{036,001,001,001,036,000,037,000,000}, 0x2287, "reflexsuperset" },
 {{000,021,021,025,021,016,000,000,000}, U(228D) }, /* union with dot */
 {{000,037,020,020,020,037,000,000,000}, U(228F) }, /* square image of */
 {{000,037,001,001,001,037,000,000,000}, U(2290) }, /* square original of */
 {{037,020,020,020,037,000,037,000,000}, U(2291) }, /* sq img or equal */
 {{037,001,001,001,037,000,037,000,000}, U(2292) }, /* sq orig or eq */
 {{000,037,021,021,021,021,000,000,000}, U(2293) }, /* square cap */
 {{000,021,021,021,021,037,000,000,000}, U(2294) }, /* square cup */
 {{000,016,025,037,025,016,000,000,000}, 0x2295, "circleplus" },
 {{000,016,021,037,021,016,000,000,000}, U(2296) }, /* circled minus (ESC) */
 {{000,016,021,025,021,016,000,000,000}, U(2299) }, /* circled dot (SI) */
 {{000,037,025,037,025,037,000,000,000}, U(229E) }, /* squared plus */
 {{000,037,021,037,021,037,000,000,000}, U(229F) }, /* squared minus (DLE) */
 {{000,037,021,025,021,037,000,000,000}, U(22A1) }, /* squared dot */
 {{000,020,020,037,020,020,000,000,000}, U(22A2) }, /* right tack */
 {{000,001,001,037,001,001,000,000,000}, U(22A3) }, /* left tack */
 {{000,037,004,004,004,004,000,000,000}, U(22A4) }, /* down tack */
 {{000,004,004,004,004,037,000,000,000}, 0x22a5, "perpendicular" },
 {{000,010,010,016,010,010,000,000,000}, U(22A6) }, /* assertion */
 {{000,010,016,010,016,010,000,000,000}, U(22A7) }, /* models */
 {{000,020,037,020,037,020,000,000,000}, U(22A8) }, /* true */
 {{000,024,024,027,024,024,000,000,000}, U(22A9) }, /* forces */
 {{000,024,027,024,027,024,000,000,000}, U(22AB) }, /* dbl v dbl rt tstile */
 {{016,021,021,021,021,021,021,021,021}, U(22C2) }, /* n-ary intersection */
 {{021,021,021,021,021,021,021,021,016}, U(22C3) }, /* n-ary union */
 {{000,004,012,021,012,004,000,000,000}, U(22C4) }, /* diamond operator */
 {{000,020,010,004,012,021,000,000,000}, U(22CB) }, /* left semidirect prod */
 {{000,001,002,004,012,021,000,000,000}, U(22CC) }, /* right semidirect prod */
 {{000,002,025,010,000,037,000,000,000}, U(22CD) }, /* reversed tilde equals */
 {{004,004,016,025,025,025,025,000,000}, U(22D4) }, /* pitchfork */
 {{007,030,003,014,020,014,003,000,000}, U(22DC) }, /* equal or less than */
 {{034,003,030,006,001,006,030,000,000}, U(22DD) }, /* equal or greater than */
 {{000,004,000,004,000,004,000,000,000}, U(22EE) }, /* vertical ellipsis */
 {{000,000,000,025,000,000,000,000,000}, U(22EF) }, /* midline ellipsis */
 {{000,001,000,004,000,020,000,000,000}, U(22F0) }, /* /-diagonal ellipsis */
 {{000,020,000,004,000,001,000,000,000}, U(22F1) }, /* \-diagonal ellipsis */
 {{037,000,017,020,037,020,017,000,000}, U(22F7) }, /* small element overbar */
 {{017,020,020,037,020,020,017,000,037}, U(22F8) }, /* element of underbar */
 {{037,000,036,001,037,001,036,000,000}, U(22FE) }, /* small contains o'bar */

 /* Miscellaneous technical */
 {{004,012,021,021,021,021,037,000,000}, 0x2302, "house" },
 {{016,010,010,010,010,010,010,000,000}, U(2308) }, /* left ceiling */
 {{016,002,002,002,002,002,002,000,000}, U(2309) }, /* right ceiling */
 {{010,010,010,010,010,010,016,000,000}, U(230A) }, /* left floor */
 {{002,002,002,002,002,002,016,000,000}, U(230B) }, /* right floor */
 {{000,000,037,020,020,000,000,000,000}, 0x2310, "revlogicalnot" },
 {{000,000,002,005,004,004,004,004,004}, 0x2320, "integraltp" },
 {{004,004,004,004,004,024,010,000,000}, 0x2321, "integralbt" },
 {{000,037,004,004,004,037,000,000,000}, U(2336) }, /* APL I-beam */
 {{016,012,012,012,012,012,016,000,000}, U(2337) }, /* APL squish quad */
 {{037,021,037,021,037,021,037,000,000}, U(2338) }, /* APL quad equal */
 {{037,025,021,037,021,025,037,000,000}, U(2339) }, /* APL quad divide */
 {{037,025,033,021,033,025,037,000,000}, U(233A) }, /* APL quad diamond */
 {{037,021,025,033,025,021,037,000,000}, U(233B) }, /* APL quad jot */
 {{004,016,025,025,025,016,004,000,000}, U(233D) }, /* APL circle stile */
 {{000,001,012,004,012,020,000,000,000}, U(233F) }, /* APL slash bar */
 {{000,020,012,004,012,001,000,000,000}, U(2340) }, /* APL backslash bar */
 {{037,021,023,025,031,021,037,000,000}, U(2341) }, /* APL quad slash */
 {{037,021,031,025,023,021,037,000,000}, U(2342) }, /* APL quad backslash */
 {{000,026,011,025,022,015,000,000,000}, U(2349) }, /* APL circle backslash */
 {{004,004,004,004,037,000,037,000,000}, U(234A) }, /* APL down tack u'bar */
 {{004,004,012,012,025,025,037,004,004}, U(234B) }, /* APL delta stile */
 {{004,016,025,025,016,004,037,000,000}, U(234E) }, /* APL down tack jot */
 {{037,000,037,004,004,004,004,000,000}, U(2351) }, /* APL up tack overbar */
 {{004,004,037,025,025,012,012,004,004}, U(2352) }, /* APL del stile */
 {{037,004,016,025,025,016,004,000,000}, U(2355) }, /* APL up tack jot */
 {{004,004,012,012,021,021,037,000,037}, U(2359) }, /* APL delta underbar */
 {{000,016,025,033,025,021,000,000,000}, U(235D) }, /* APL up shoe jot */
 {{037,025,025,021,021,021,037,000,000}, U(235E) }, /* APL quote quad */
 {{016,025,037,025,033,021,016,000,000}, U(235F) }, /* APL circle star */
 {{037,021,025,021,025,021,037,000,000}, U(2360) }, /* APL quad colon */
 {{012,000,025,016,004,016,025,000,000}, U(2363) }, /* APL star dieresis */
 {{012,000,004,012,012,004,000,000,000}, U(2364) }, /* APL jot dieresis */
 {{012,000,010,025,002,000,000,000,000}, U(2368) }, /* APL tilde dieresis */
 {{000,000,000,037,000,000,004,004,010}, U(236A) }, /* APL comma bar */
 {{037,021,021,016,033,004,004,000,000}, U(236B) }, /* APL del tilde */
 {{004,012,021,037,021,012,004,000,000}, U(236C) }, /* APL zilde */
 {{010,025,002,000,021,012,004,000,000}, U(2371) }, /* APL nor */
 {{010,025,002,000,004,012,021,000,000}, U(2372) }, /* APL nand */
 {{000,000,014,004,004,004,002,000,000}, U(2373) }, /* APL iota */
 {{000,000,016,021,021,021,036,020,020}, U(2374) }, /* APL rho */
 {{000,000,012,021,025,025,012,000,000}, U(2375) }, /* APL omega */
 {{000,000,015,022,022,022,015,000,037}, U(2376) }, /* APL alpha underbar */
 {{017,020,037,020,017,000,037,000,000}, U(2377) }, /* APL epsilon underbar */
 {{000,000,014,004,004,004,002,000,037}, U(2378) }, /* APL iota underbar */
 {{000,000,012,021,025,025,012,000,037}, U(2379) }, /* APL omega underbar*/
 {{000,000,015,022,022,022,015,000,000}, U(237A) }, /* APL alpha */
 {{000,000,005,002,025,010,000,000,000}, U(237B) }, /* crossed tick (NAK) */
 {{000,000,000,000,000,033,016,000,000}, U(237D) }, /* nbsp symbol */
 {{000,016,021,037,012,033,000,000,000}, U(237E) }, /* bell symbol (BEL) */
 {{004,004,004,012,004,004,004,000,000}, U(237F) }, /* vert line w/dot (EOM) */
 {{000,016,012,012,012,033,000,000,000}, U(238D) }, /* monostable (SYN) */
 {{000,007,012,012,012,034,000,000,000}, U(238E) }, /* hysteresis */
 {{000,000,037,000,025,000,000,000,000}, U(2393) }, /* directcurrent */
 {{037,021,021,021,021,021,037,000,000}, U(2395) }, /* APL quad */
 {{002,004,010,010,010,010,010,010,010}, U(239B) }, /* long parenleft top */
 {{010,010,010,010,010,010,010,010,010}, U(239C) }, /* long parenleft middle */
 {{010,010,010,010,010,004,002,000,000}, U(239D) }, /* long parenleft bottom */
 {{010,004,002,002,002,002,002,002,002}, U(239E) }, /* long parenright top */
 {{002,002,002,002,002,002,002,002,002}, U(239F) }, /* long parenright middle */
 {{002,002,002,002,002,004,010,000,000}, U(23A0) }, /* long parenright bottom */
 {{017,010,010,010,010,010,010,010,010}, U(23A1) }, /* long bracketleft top */
 {{010,010,010,010,010,010,010,010,010}, U(23A2) }, /* long bracketleft mid */
 {{010,010,010,010,010,010,017,000,000}, U(23A3) }, /* long bracketleft bot */
 {{036,002,002,002,002,002,002,002,002}, U(23A4) }, /* long bracketright top */
 {{002,002,002,002,002,002,002,002,002}, U(23A5) }, /* long bracketright mid */
 {{002,002,002,002,002,002,036,000,000}, U(23A6) }, /* long bracketright bot */
 {{003,004,004,004,004,004,004,004,004}, U(23A7) }, /* long braceleft top */
 {{004,004,004,010,004,004,004,004,004}, U(23A8) }, /* long braceleft middle */
 {{004,004,004,004,004,004,003,000,000}, U(23A9) }, /* long braceleft bottom */
 {{004,004,004,004,004,004,004,004,004}, U(23AA) }, /* long brace extension */
 {{030,004,004,004,004,004,004,004,004}, U(23AB) }, /* long braceright top */
 {{004,004,004,002,004,004,004,004,004}, U(23AC) }, /* long braceright middle */
 {{004,004,004,004,004,004,030,000,000}, U(23AD) }, /* long braceright bottom */
 {{004,004,004,004,004,004,004,004,004}, U(23AE) }, /* integral extension */
 {{003,004,004,004,004,004,004,004,030}, U(23B0) }, /* two-level brace / */
 {{030,004,004,004,004,004,004,004,003}, U(23B1) }, /* two-level brace \ */
 {{037,020,010,010,004,004,002,002,001}, U(23B2) }, /* summation top */
 {{001,002,002,004,004,010,010,020,037}, U(23B3) }, /* summation bottom */
 {{037,000,000,000,000,000,000,000,000}, U(23BA) }, /* horizontal scan 1 */
 {{000,000,037,000,000,000,000,000,000}, U(23BB) }, /* horizontal scan 3 */
 {{000,000,000,000,000,000,037,000,000}, U(23BC) }, /* horizontal scan 7 */
 {{000,000,000,000,000,000,000,000,037}, U(23BD) }, /* horizontal scan 9 */
 {{004,004,037,000,016,000,004,000,000}, U(23DA) }, /* earth */

 /* Control pictures */
 {{032,026,022,022,000,011,011,011,006}, U(2400) }, /* NUL */
 {{010,020,010,020,005,005,007,005,005}, U(2401) }, /* SOH */
 {{010,020,010,020,005,005,002,005,005}, U(2402) }, /* STX */
 {{034,020,030,020,035,005,002,005,005}, U(2403) }, /* ETX */
 {{034,020,030,020,037,002,002,002,002}, U(2404) }, /* EOT */
 {{034,020,030,022,035,005,005,002,001}, U(2405) }, /* ENQ */
 {{010,024,034,024,025,005,006,005,005}, U(2406) }, /* ACK */
 {{030,024,030,024,030,002,002,002,003}, U(2407) }, /* BEL */
 {{030,024,030,024,030,001,002,001,002}, U(2408) }, /* BS */
 {{024,024,034,024,027,002,002,002,002}, U(2409) }, /* HT */
 {{020,020,020,020,037,004,006,004,004}, U(240A) }, /* LF */
 {{024,024,024,010,017,002,002,002,002}, U(240B) }, /* VT */
 {{034,020,030,020,027,004,006,004,004}, U(240C) }, /* FF */
 {{014,020,020,020,016,005,006,005,005}, U(240D) }, /* CR */
 {{014,020,010,004,030,002,005,005,002}, U(240E) }, /* SO */
 {{014,020,010,004,031,001,001,001,001}, U(240F) }, /* SI */
 {{030,024,024,024,030,002,002,002,003}, U(2410) }, /* DLE */
 {{030,024,024,024,031,001,001,001,001}, U(2411) }, /* DC1 */
 {{030,024,024,024,030,002,001,002,003}, U(2412) }, /* DC2 */
 {{030,024,024,030,002,001,002,001,002}, U(2413) }, /* DC3 */
 {{030,024,024,024,031,003,005,007,001}, U(2414) }, /* DC4 */
 {{032,026,022,022,000,005,006,005,005}, U(2415) }, /* NAK */
 {{010,020,010,020,005,005,002,002,002}, U(2416) }, /* SYN */
 {{034,020,030,020,036,005,006,005,006}, U(2417) }, /* ETB */
 {{014,020,020,014,000,015,013,011,011}, U(2418) }, /* CAN */
 {{034,020,030,020,034,000,033,025,021}, U(2419) }, /* EOM */
 {{010,020,010,020,006,005,006,005,006}, U(241A) }, /* SUB */
 {{030,020,030,020,030,003,004,004,003}, U(241B) }, /* ESC */
 {{034,020,030,020,023,004,002,001,006}, U(241C) }, /* FS */
 {{014,020,020,024,014,001,002,001,002}, U(241D) }, /* GS */
 {{030,024,030,024,024,001,002,001,002}, U(241E) }, /* RS */
 {{022,022,014,000,003,004,002,001,006}, U(241F) }, /* US */
 {{010,020,010,020,006,005,006,004,004}, U(2420) }, /* SP */
 {{030,024,024,030,000,007,002,002,002}, U(2421) }, /* DEL */
 {{000,000,000,000,000,021,037,000,000}, U(2423) }, /* Visible space */
 {{032,026,022,022,000,004,004,004,007}, U(2424) }, /* NL */
 {{004,011,022,004,011,022,004,000,000}, U(2425) }, /* pictorial DEL */
 {{016,021,010,004,004,000,004,000,000}, U(2426) }, /* pictorial SUB */

 /* Box drawing */
 {{000,000,000,000,037,000,000,000,000}, 0x2500, "SF100000" },
 {{000,000,000,037,037,037,000,000,000}, U(2501) },
 {{004,004,004,004,004,004,004,004,004}, 0x2502, "SF110000" },
 {{016,016,016,016,016,016,016,016,016}, U(2503) },
 {{000,000,000,000,025,000,000,000,000}, U(2504) },
 {{000,000,000,025,025,025,000,000,000}, U(2505) },
 {{004,004,000,004,004,004,000,004,004}, U(2506) },
 {{016,016,000,016,016,016,000,016,016}, U(2507) },
 {{004,000,004,004,000,004,000,004,004}, U(250A) },
 {{016,000,016,016,000,016,000,016,016}, U(250B) },
 {{000,000,000,000,007,004,004,004,004}, 0x250c, "SF010000" },
 {{000,000,000,007,007,007,004,004,004}, U(250D) },
 {{000,000,000,000,017,016,016,016,016}, U(250E) },
 {{000,000,000,017,017,017,016,016,016}, U(250F) },
 {{000,000,000,000,034,004,004,004,004}, 0x2510, "SF030000" },
 {{000,000,000,034,034,034,004,004,004}, U(2511) },
 {{000,000,000,000,036,016,016,016,016}, U(2512) },
 {{000,000,000,036,036,036,016,016,016}, U(2513) },
 {{004,004,004,004,007,000,000,000,000}, 0x2514, "SF020000" },
 {{004,004,004,007,007,007,000,000,000}, U(2515) },
 {{016,016,016,016,017,000,000,000,000}, U(2516) },
 {{016,016,016,017,017,017,000,000,000}, U(2517) },
 {{004,004,004,004,034,000,000,000,000}, 0x2518, "SF040000" },
 {{004,004,004,034,034,034,000,000,000}, U(2519) },
 {{016,016,016,016,036,000,000,000,000}, U(251A) },
 {{016,016,016,036,036,036,000,000,000}, U(251B) },
 {{004,004,004,004,007,004,004,004,004}, 0x251c, "SF080000" },
 {{004,004,004,007,007,007,004,004,004}, U(251D) },
 {{016,016,016,016,017,004,004,004,004}, U(251E) },
 {{004,004,004,004,017,016,016,016,016}, U(251F) },
 {{016,016,016,016,017,016,016,016,016}, U(2520) },
 {{016,016,016,017,017,017,004,004,004}, U(2521) },
 {{004,004,004,017,017,017,016,016,016}, U(2522) },
 {{016,016,016,017,017,017,016,016,016}, U(2523) },
 {{004,004,004,004,034,004,004,004,004}, 0x2524, "SF090000" },
 {{004,004,004,034,034,034,004,004,004}, U(2525) },
 {{016,016,016,016,036,004,004,004,004}, U(2526) },
 {{004,004,004,004,036,016,016,016,016}, U(2527) },
 {{016,016,016,016,036,016,016,016,016}, U(2528) },
 {{016,016,016,036,036,036,004,004,004}, U(2529) },
 {{004,004,004,036,036,036,016,016,016}, U(252A) },
 {{016,016,016,036,036,036,016,016,016}, U(252B) },
 {{000,000,000,000,037,004,004,004,004}, 0x252c, "SF060000" },
 {{000,000,000,034,037,034,004,004,004}, U(252D) },
 {{000,000,000,007,037,007,004,004,004}, U(252E) },
 {{000,000,000,037,037,037,004,004,004}, U(252F) },
 {{000,000,000,000,037,016,016,016,016}, U(2530) },
 {{000,000,000,036,037,036,016,016,016}, U(2531) },
 {{000,000,000,017,037,017,016,016,016}, U(2532) },
 {{000,000,000,037,037,037,016,016,016}, U(2533) },
 {{004,004,004,004,037,000,000,000,000}, 0x2534, "SF070000" },
 {{004,004,004,034,037,034,000,000,000}, U(2535) },
 {{004,004,004,007,037,007,000,000,000}, U(2536) },
 {{004,004,004,037,037,037,000,000,000}, U(2537) },
 {{016,016,016,016,037,000,000,000,000}, U(2538) },
 {{016,016,016,036,037,036,000,000,000}, U(2539) },
 {{016,016,016,017,037,017,000,000,000}, U(253A) },
 {{016,016,016,037,037,037,000,000,000}, U(253B) },
 {{004,004,004,004,037,004,004,004,004}, 0x253c, "SF050000" },
 {{004,004,004,034,037,034,004,004,004}, U(253D) },
 {{004,004,004,007,037,007,004,004,004}, U(253E) },
 {{004,004,004,037,037,037,004,004,004}, U(253F) },
 {{016,016,016,016,037,004,004,004,004}, U(2540) },
 {{004,004,004,004,037,016,016,016,016}, U(2541) },
 {{016,016,016,016,037,016,016,016,016}, U(2542) },
 {{016,016,016,036,037,036,004,004,004}, U(2543) },
 {{016,016,016,017,037,017,004,004,004}, U(2544) },
 {{004,004,004,036,037,036,016,016,016}, U(2545) },
 {{004,004,004,017,037,017,016,016,016}, U(2546) },
 {{016,016,016,037,037,037,004,004,004}, U(2547) },
 {{004,004,004,037,037,037,016,016,016}, U(2548) },
 {{016,016,016,036,037,036,016,016,016}, U(2549) },
 {{016,016,016,017,037,017,016,016,016}, U(254A) },
 {{016,016,016,037,037,037,016,016,016}, U(254B) },
 {{000,000,000,000,033,000,000,000,000}, U(254C) },
 {{000,000,000,033,033,033,000,000,000}, U(254D) },
 {{004,004,004,004,000,004,004,004,004}, U(254E) },
 {{016,016,016,016,000,016,016,016,016}, U(254F) },
 {{000,000,000,037,000,037,000,000,000}, 0x2550, "SF430000" },
 {{012,012,012,012,012,012,012,012,012}, 0x2551, "SF240000" },
 {{000,000,000,007,004,007,004,004,004}, 0x2552, "SF510000" },
 {{000,000,000,000,017,012,012,012,012}, 0x2553, "SF520000" },
 {{000,000,000,017,010,013,012,012,012}, 0x2554, "SF390000" },
 {{000,000,000,034,004,034,004,004,004}, 0x2555, "SF220000" },
 {{000,000,000,000,036,012,012,012,012}, 0x2556, "SF210000" },
 {{000,000,000,036,002,032,012,012,012}, 0x2557, "SF250000" },
 {{004,004,004,007,004,007,000,000,000}, 0x2558, "SF500000" },
 {{012,012,012,012,017,000,000,000,000}, 0x2559, "SF490000" },
 {{012,012,012,013,010,017,000,000,000}, 0x255a, "SF380000" },
 {{004,004,004,034,004,034,000,000,000}, 0x255b, "SF280000" },
 {{012,012,012,012,036,000,000,000,000}, 0x255c, "SF270000" },
 {{012,012,012,032,002,036,000,000,000}, 0x255d, "SF260000" },
 {{004,004,004,007,004,007,004,004,004}, 0x255e, "SF360000" },
 {{012,012,012,012,013,012,012,012,012}, 0x255f, "SF370000" },
 {{012,012,012,013,010,013,012,012,012}, 0x2560, "SF420000" },
 {{004,004,004,034,004,034,004,004,004}, 0x2561, "SF190000" },
 {{012,012,012,012,032,012,012,012,012}, 0x2562, "SF200000" },
 {{012,012,012,032,002,032,012,012,012}, 0x2563, "SF230000" },
 {{000,000,000,037,000,037,004,004,004}, 0x2564, "SF470000" },
 {{000,000,000,000,037,012,012,012,012}, 0x2565, "SF480000" },
 {{000,000,000,037,000,033,012,012,012}, 0x2566, "SF410000" },
 {{004,004,004,037,000,037,000,000,000}, 0x2567, "SF450000" },
 {{012,012,012,012,037,000,000,000,000}, 0x2568, "SF460000" },
 {{012,012,012,033,000,037,000,000,000}, 0x2569, "SF400000" },
 {{004,004,004,037,004,037,004,004,004}, 0x256a, "SF540000" },
 {{012,012,012,012,037,012,012,012,012}, 0x256b, "SF530000" },
 {{012,012,012,033,000,033,012,012,012}, 0x256c, "SF440000" },
 {{000,000,000,000,003,004,004,004,004}, U(256D) },
 {{000,000,000,000,030,004,004,004,004}, U(256E) },
 {{004,004,004,004,030,000,000,000,000}, U(256F) },
 {{004,004,004,004,003,000,000,000,000}, U(2570) },
 {{000,000,000,000,034,000,000,000,000}, U(2574) },
 {{004,004,004,004,004,000,000,000,000}, U(2575) },
 {{000,000,000,000,007,000,000,000,000}, U(2576) },
 {{000,000,000,000,004,004,004,004,004}, U(2577) },
 {{000,000,000,034,034,034,000,000,000}, U(2578) },
 {{016,016,016,016,016,000,000,000,000}, U(2579) },
 {{000,000,000,007,007,007,000,000,000}, U(257A) },
 {{000,000,000,000,016,016,016,016,016}, U(257B) },
 {{000,000,000,007,037,007,000,000,000}, U(257C) },
 {{004,004,004,004,016,016,016,016,016}, U(257D) },
 {{000,000,000,034,037,034,000,000,000}, U(257E) },
 {{016,016,016,016,016,004,004,004,004}, U(257F) },

 /* Block elements */
 {{0x03}, 0x2580, "upblock", MOS4 }, {{0x03}, -1, "upblock.sep4", SEP4 },
 {{0x0c}, 0x2584, "dnblock", MOS4 }, {{0x0c}, -1, "dnblock.sep4", SEP4 },
 {{0x0f}, 0x2588, "block",   MOS4 }, {{0x0f}, -1, "block.sep4",   SEP4 },
 {{0x3f}, -1, "block.sep6", SEP6 },
 {{0x05}, 0x258c, "lfblock", MOS4 }, {{0x05}, -1, "lfblock.sep4", SEP4 },
 {{0x15}, -1, "lfblock.sep6", SEP6 },
 {{0x0a}, 0x2590, "rtblock", MOS4 }, {{0x0a}, -1, "rtblock.sep4", SEP4 },
 {{0x2a}, -1, "rtblock.sep6", SEP6 },
 {{025,000,012,000,025,000,012,000,025}, 0x2591, "ltshade" },
 {{022,011,004,022,011,004,022,011,004}, 0x2592, "shade" },
 {{025,037,012,037,025,037,012,037,025}, 0x2593, "dkshade" },
#define M(x, u) { {x}, U(u), MOS4 }, { {x}, -1, "uni" #u ".sep4", SEP4 }
 M( 4, 2596), M( 8, 2597), M( 1, 2598), M(13, 2599),
 M( 9, 259A), M( 7, 259B), M(11, 259C), M( 2, 259D),
 M( 6, 259E), M(14, 259F),
#undef M

 /* Geometric shapes */
 {{037,021,021,021,021,021,037,000,000}, 0x25a1, "H22073" },
 {{000,000,016,016,016,000,000,000,000}, 0x25aa, "H18543" },
 {{000,000,016,012,016,000,000,000,000}, 0x25ab, "H18551" },
 {{000,000,037,037,037,000,000,000,000}, 0x25ac, "filledrect" },
 {{000,000,037,021,037,000,000,000,000}, U(25AD) }, /* rect */
 {{004,004,016,016,037,037,037,000,000}, 0x25b2, "triagup" },
 {{010,014,016,017,016,014,010,000,000}, U(25B6) }, /* blacktriangleright */
 {{030,024,022,021,022,024,030,000,000}, U(25B7) }, /* whitetriangleright */
 {{000,020,034,037,034,020,000,000,000}, 0x25ba, "triagrt" },
 {{037,037,037,016,016,004,004,000,000}, 0x25bc, "triagdn" },
 {{001,003,007,017,007,003,001,000,000}, U(25C0) }, /* blacktriangleleft */
 {{003,005,011,021,011,005,003,000,000}, U(25C1) }, /* whitetriangleleft */
 {{000,001,007,037,007,001,000,000,000}, 0x25c4, "triaglf" },
 {{000,004,016,037,016,004,000,000,000}, U(25C6) }, /* black diamond shape */
 {{000,004,012,021,012,004,000,000,000}, U(25C7) }, /* white diamond shape */
 {{004,012,012,021,012,012,004,000,000}, 0x25ca, "lozenge" },
 {{000,016,021,021,021,016,000,000,000}, 0x25cb, "circle" },
 {{000,004,021,000,021,004,000,000,000}, U(25CC) }, /* dottedcircle */
 {{000,016,037,037,037,016,000,000,000}, 0x25cf, "H18533" },
 {{037,037,021,021,021,037,037,000,000}, 0x25d8, "invbullet" },
 {{037,021,004,016,004,021,037,000,000}, 0x25d9, "invcircle" },
 {{000,037,025,025,025,037,000,000,000}, U(25EB) }, /* squared vertical line */
 {{000,037,025,035,021,037,000,000,000}, U(25F0) }, /* square upper left quad */
 {{000,037,021,035,025,037,000,000,000}, U(25F1) }, /* square lower left quad */
 {{000,037,025,027,021,037,000,000,000}, U(25F2) }, /* square lower right quad */
 {{000,037,021,027,025,037,000,000,000}, U(25F3) }, /* square upper right quad */
 {{000,016,025,035,021,016,000,000,000}, U(25F4) }, /* circle upper left quad */
 {{000,016,021,035,025,016,000,000,000}, U(25F5) }, /* circle lower left quad */
 {{000,016,025,027,021,016,000,000,000}, U(25F6) }, /* circle lower right quad */
 {{000,016,021,027,025,016,000,000,000}, U(25F7) }, /* circle upper right quad */

 /* Miscellaneous symbols */
 {{001,002,004,010,005,003,007,000,000}, U(2607) }, /* lightning */
 {{037,022,024,030,025,023,027,000,000}, U(2608) }, /* thunderstorm */
 {{000,016,021,025,021,016,000,000,000}, U(2609) }, /* astrological sun */
 {{000,012,000,000,016,021,000,000,000}, U(2639) }, /* frownface */
 {{000,012,000,000,021,016,000,000,000}, 0x263a, "smileface" },
 {{016,037,025,037,037,021,016,000,000}, 0x263b, "invsmileface" },
 {{000,025,016,033,016,025,000,000,000}, 0x263c, "sun" },
 {{016,021,021,016,004,016,004,000,000}, 0x2640, "female" },
 {{007,003,005,014,022,022,014,000,000}, 0x2642, "male" },
 {{000,004,016,037,037,004,016,000,000}, 0x2660, "spade" },
 {{000,016,016,037,037,004,016,000,000}, 0x2663, "club" },
 {{000,012,037,037,016,004,000,000,000}, 0x2665, "heart" },
 {{000,004,016,037,016,004,000,000,000}, 0x2666, "diamond" },
 {{002,002,002,002,006,016,004,000,000}, U(2669) }, /* crotchet */
 {{004,006,005,005,014,034,010,000,000}, 0x266a, "musicalnote" },
 {{003,015,011,011,013,033,030,000,000}, 0x266b, "musicalnotedbl" },
 {{017,011,017,011,011,033,033,000,000}, U(266C) }, /* semiquavers */
 {{010,010,013,015,011,012,014,000,000}, U(266D) }, /* flat */
 {{010,013,015,011,013,015,001,000,000}, U(266E) }, /* natural */
 {{002,013,016,033,016,032,010,000,000}, U(266F) }, /* sharp */

 /* Dingbats */
 {{000,004,022,017,022,004,000,000,000}, U(2708) }, /* airplane */
 {{000,000,001,002,024,010,000,000,000}, U(2713) }, /* tick (ACK) */

 /* Miscellaneous mathematical symbols-A */
 {{004,004,004,016,004,004,004,000,000}, U(27CA) }, /* vert bar horiz stroke */
 {{000,025,025,025,025,016,000,000,000}, U(27D2) }, /* upward element of */

 /* Supplemental arrows-B */
 {{000,037,016,025,004,004,000,000,000}, U(2912) }, /* arrowupbar */
 {{000,004,004,025,016,037,000,000,000}, U(2913) }, /* arrowdownbar */
 {{000,004,016,025,004,030,000,000,000}, U(2934) }, /* arrow right curve up */
 {{000,030,004,025,016,004,000,000,000}, U(2935) }, /* arrow right curve down */
 {{001,005,011,036,010,004,000,000,000}, U(2936) }, /* arrow down curve left */
 {{020,024,022,017,002,004,000,000,000}, U(2937) }, /* arrow down curve right */
 {{000,024,030,037,020,020,000,000,000}, U(2952) }, /* harpoons to bars... */
 {{000,005,003,037,001,001,000,000,000}, U(2953) },
 {{000,037,006,005,004,004,000,000,000}, U(2954) },
 {{000,004,004,005,006,037,000,000,000}, U(2955) },
 {{000,020,020,037,030,024,000,000,000}, U(2956) },
 {{000,001,001,037,003,005,000,000,000}, U(2957) },
 {{000,037,014,024,004,004,000,000,000}, U(2958) },
 {{000,004,004,024,014,037,000,000,000}, U(2959) },
 {{000,005,011,037,001,001,000,000,000}, U(295A) }, /* ... and from bars */
 {{000,024,022,037,020,020,000,000,000}, U(295B) },
 {{000,004,006,005,004,037,000,000,000}, U(295C) },
 {{000,037,004,005,006,004,000,000,000}, U(295D) },
 {{000,001,001,037,011,005,000,000,000}, U(295E) },
 {{000,020,020,037,022,024,000,000,000}, U(295F) },
 {{000,004,014,024,004,037,000,000,000}, U(2960) },
 {{000,037,004,024,014,004,000,000,000}, U(2961) },

 /* Miscellaneous mathematical symbols-B */
 {{025,025,025,025,025,025,025,000,000}, U(2980) }, /* triple vertical bar */
 {{000,016,025,025,025,016,000,000,000}, U(29B6) }, /* circled vert bar */
 {{000,016,025,037,021,016,000,000,000}, U(29BA) }, /* circled horiz up */

 /* Supplemental mathematical operators */
 {{004,004,037,004,004,000,004,000,000}, U(2A25) }, /* plus dot */
 {{000,000,000,037,000,004,000,000,000}, U(2A2A) }, /* minus dot */
 {{000,020,000,037,000,001,000,000,000}, U(2A2B) }, /* minus falling dots */
 {{000,001,000,037,000,020,000,000,000}, U(2A2C) }, /* minus rising dots */
 {{004,021,012,004,012,021,000,000,000}, U(2A30) }, /* dot times */
 {{000,016,021,025,021,021,000,000,000}, U(2A40) }, /* intersection with dot */
 {{004,000,004,012,021,000,000,000,000}, U(2A51) }, /* dot and */
 {{004,000,021,012,004,000,000,000,000}, U(2A52) }, /* dot or */
 {{000,000,037,000,037,000,004,000,000}, U(2A66) }, /* equals dot */
 {{004,000,037,000,037,000,037,000,000}, U(2A67) }, /* dot identical */
 {{004,000,010,025,002,000,000,000,000}, U(2A6A) }, /* dot tilde */
 {{000,001,010,025,002,020,000,000,000}, U(2A6B) }, /* tilde rising dots */
 {{037,000,037,000,010,025,002,000,000}, U(2A73) }, /* equals tilde */
 {{012,000,037,000,037,000,012,000,000}, U(2A77) }, /* equals with four dots */
 {{004,033,000,037,000,037,000,000,000}, U(2AAE) }, /* bumpy equals */
 {{000,017,020,024,020,017,000,000,000}, U(2ABD) }, /* subset with dot */
 {{000,036,001,005,001,036,000,000,000}, U(2ABE) }, /* superset with dot */
 {{017,020,017,000,004,016,004,000,000}, U(2ABF) }, /* subset plus */
 {{036,001,036,000,004,016,004,000,000}, U(2AC0) }, /* superset plus */
 {{017,020,017,000,012,004,012,000,000}, U(2AC1) }, /* subset times */
 {{036,001,036,000,012,004,012,000,000}, U(2AC2) }, /* superset times */
 {{004,000,017,020,017,000,037,000,000}, U(2AC3) }, /* dot subset or equal */
 {{004,000,036,001,036,000,037,000,000}, U(2AC4) }, /* dot superset or equal */
 {{017,020,017,000,037,000,037,000,000}, U(2AC5) }, /* subset equals */
 {{036,001,036,000,037,000,037,000,000}, U(2AC6) }, /* superset equals */
 {{017,020,017,000,010,025,002,000,000}, U(2AC7) }, /* subset tilde */
 {{036,001,036,000,010,025,002,000,000}, U(2AC8) }, /* superset tilde */
 {{000,017,021,021,021,017,000,000,000}, U(2ACF) }, /* closed subset */
 {{000,036,021,021,021,036,000,000,000}, U(2AD0) }, /* closed superset */
 {{017,021,021,021,017,000,037,000,000}, U(2AD1) }, /* closed subset or eq */
 {{036,021,021,021,036,000,037,000,000}, U(2AD2) }, /* closed superset or eq */
 {{017,020,017,000,036,001,036,000,000}, U(2AD3) }, /* subset superset */
 {{036,001,036,000,017,020,017,000,000}, U(2AD4) }, /* superset subset */
 {{017,020,017,000,017,020,017,000,000}, U(2AD5) }, /* subset subset */
 {{036,001,036,000,036,001,036,000,000}, U(2AD6) }, /* superset superset */
 {{000,016,025,025,025,025,000,000,000}, U(2AD9) }, /* downward element of */
 {{016,004,016,025,025,025,025,000,000}, U(2ADA) }, /* tee top pitchfork */
 {{004,016,025,025,025,025,004,000,000}, U(2ADB) }, /* transversal i'section */
 {{000,002,002,016,002,002,000,000,000}, U(2ADE) }, /* short left tack */
 {{000,000,037,004,004,000,000,000,000}, U(2ADF) }, /* short down tack */
 {{000,000,004,004,037,000,000,000,000}, U(2AE0) }, /* short up tack */
 {{020,037,020,037,020,037,020,000,000}, U(2AE2) }, /* triple rt turnstile */
 {{000,005,005,035,005,005,000,000,000}, U(2AE3) }, /* dbl vert left tstile */
 {{000,001,037,001,037,001,000,000,000}, U(2AE4) }, /* dbl left turnstile */
 {{000,005,035,005,035,005,000,000,000}, U(2AE5) }, /* dbl v dbl left tstile */
 {{000,024,024,037,024,024,000,000,000}, U(2AE6) },
 {{000,037,000,037,004,004,000,000,000}, U(2AE7) }, /* short down tack o'bar */
 {{000,004,004,037,000,037,000,000,000}, U(2AE8) }, /* short up tack u'bar */
 {{004,004,037,000,037,004,004,000,000}, U(2AE9) }, /* short up down tacks */
 {{000,037,012,012,012,012,000,000,000}, U(2AEA) }, /* dbl down tack */
 {{000,012,012,012,012,037,000,000,000}, U(2AEB) }, /* dbl up tack */
 {{000,037,001,037,001,001,000,000,000}, U(2AEC) }, /* dbl not sign */
 {{000,037,020,037,020,020,000,000,000}, U(2AED) }, /* dbl reversed not */
 {{012,012,012,037,012,012,012,000,000}, U(2AF2) }, /* dbl v bar h stroke */
 {{025,025,025,025,025,025,025,000,000}, U(2AF4) }, /* triple v bar operator */
 {{025,025,025,037,025,025,025,000,000}, U(2AF5) }, /* triple v bar h stroke */
 {{025,025,025,025,025,025,025,025,025}, U(2AFC) }, /* large triple vert bar */

 /* Miscellaneous symbols and arrows */
 {{000,025,000,021,000,025,000,000,000}, U(2B1A) }, /* dottedsquare */

 /* Latin extended-C */
 {{021,021,037,021,021,012,004,000,000}, U(2C6F) }, /* turned A */
 {{000,000,022,025,024,024,010,000,000}, U(2C71) }, /* vhook */
 {{000,000,037,001,017,001,037,000,000}, U(2C7B) }, /* small cap turned E */
 {{016,021,020,016,001,021,016,004,003}, U(2C7E) }, /* S with swash tail */
 {{037,001,002,004,010,020,030,004,003}, U(2C7F) }, /* Z with swash tail */

 /* Supplemental punctuation */
 {{000,000,004,000,004,014,024,025,016}, U(2E18) }, /* gnaborretni */
 {{011,022,022,022,022,022,011,000,000}, U(2E28) }, /* left double paren */
 {{022,011,011,011,011,011,022,000,000}, U(2E29) }, /* right double paren */
 {{000,000,021,000,000,000,004,000,000}, U(2E2A) }, /* two dots over one */
 {{000,000,004,000,000,000,021,000,000}, U(2E2B) }, /* one dot over two */
 {{000,000,021,000,000,000,021,000,000}, U(2E2C) }, /* squared four dots */
 {{000,000,004,000,025,000,004,000,000}, U(2E2D) }, /* five dot mark */
 {{016,021,010,004,004,000,004,000,000}, U(2E2E) }, /* reversed question */
 {{004,004,034,004,004,004,004,004,004}, U(2E36) }, /* dagger w/left guard */
 {{004,004,007,004,004,004,004,004,004}, U(2E37) }, /* dagger w/right guard */
 {{004,004,004,004,004,004,037,004,004}, U(2E38) }, /* turned dagger */
 {{000,000,016,000,016,000,000,000,000}, U(2E40) }, /* double hyphen */
 {{000,000,000,000,000,004,004,002,000}, U(2E41) }, /* reversed comma */
 {{000,000,000,000,000,022,022,011,000}, U(2E42) }, /* dbl low rev9 quote */
 {{004,004,037,004,037,004,037,004,004}, U(2E4B) }, /* triple dagger */

 /* Latin extended-D */
 {{000,000,037,020,036,020,020,000,000}, U(A730) }, /* small cap F */
 {{000,000,017,020,016,001,036,000,000}, U(A731) }, /* small cap S */
 {{037,001,001,001,001,001,001,000,000}, U(A780) }, /* turned L */
 {{016,004,004,004,004,004,006,000,000}, U(A781) }, /* turned l */
 {{000,000,016,021,025,022,015,000,000}, U(A7AF) }, /* small cap Q */
 {{021,011,005,003,005,011,021,000,000}, U(A7B0) }, /* turned K */
 {{004,004,004,004,004,004,037,000,000}, U(A7B1) }, /* turned T */
 {{037,001,001,017,001,001,001,000,000}, U(A7FB) }, /* reversed F */
 {{017,021,021,017,001,001,001,000,000}, U(A7FC) }, /* reversed P */
 {{021,021,021,025,025,033,021,000,000}, U(A7FD) }, /* inverted M */

 /* Private use area */
 /* U+EE00--U+EE7F: zvbi mosaic graphics */
#define M(x) {{(0x##x & 0x1f)|((0x##x & 0x40)>>1)}, U(EE##x), \
	      (0x##x & 0x20 ? SEP6 : MOS6)}
 M(00), M(01), M(02), M(03), M(04), M(05), M(06), M(07),
 M(08), M(09), M(0A), M(0B), M(0C), M(0D), M(0E), M(0F),
 M(10), M(11), M(12), M(13), M(14), M(15), M(16), M(17),
 M(18), M(19), M(1A), M(1B), M(1C), M(1D), M(1E), M(1F),
 M(20), M(21), M(22), M(23), M(24), M(25), M(26), M(27),
 M(28), M(29), M(2A), M(2B), M(2C), M(2D), M(2E), M(2F),
 M(30), M(31), M(32), M(33), M(34), M(35), M(36), M(37),
 M(38), M(39), M(3A), M(3B), M(3C), M(3D), M(3E), M(3F),
 M(40), M(41), M(42), M(43), M(44), M(45), M(46), M(47),
 M(48), M(49), M(4A), M(4B), M(4C), M(4D), M(4E), M(4F),
 M(50), M(51), M(52), M(53), M(54), M(55), M(56), M(57),
 M(58), M(59), M(5A), M(5B), M(5C), M(5D), M(5E), M(5F),
 M(60), M(61), M(62), M(63), M(64), M(65), M(66), M(67),
 M(68), M(69), M(6A), M(6B), M(6C), M(6D), M(6E), M(6F),
 M(70), M(71), M(72), M(73), M(74), M(75), M(76), M(77),
 M(78), M(79), M(7A), M(7B), M(7C), M(7D), M(7E), M(7F),
#undef M

 /* Alphabetic presentation forms */
 {{006,010,036,012,012,012,017,000,000}, 0xfb01, "fi" },
 {{006,012,012,036,012,012,017,000,000}, 0xfb02, "fl" },

 /* Specials */
 {{016,021,025,011,016,012,016,000,000}, U(FFFD) }, /* replacement */

 /* Shavian */
 {{030,004,004,004,004,004,004,000,000}, U(10450) }, /* peep */
 {{014,024,004,004,004,004,004,000,000}, U(10451) }, /* tot */
 {{004,004,004,014,020,020,014,000,000}, U(10452) }, /* kick */
 {{004,004,004,004,004,004,030,000,000}, U(10453) }, /* fee */
 {{014,002,001,017,021,021,016,000,000}, U(10454) }, /* thigh */
 {{017,020,016,001,001,001,036,000,000}, U(10455) }, /* so */
 {{001,002,004,010,020,020,016,000,000}, U(10456) }, /* sure */
 {{015,022,004,010,020,020,016,000,000}, U(10457) }, /* church */
 {{020,020,010,004,002,001,001,000,000}, U(10458) }, /* yea */
 {{004,012,021,012,004,012,021,000,000}, U(10459) }, /* hung */
 {{000,000,004,004,004,004,004,004,003}, U(1045A) }, /* bib */
 {{000,000,004,004,004,004,004,005,006}, U(1045B) }, /* dead */
 {{000,000,006,001,001,006,004,004,004}, U(1045C) }, /* gag */
 {{000,000,003,004,004,004,004,004,004}, U(1045D) }, /* vow */
 {{000,000,016,021,021,036,020,010,006}, U(1045E) }, /* they */
 {{000,000,036,001,001,001,016,020,017}, U(1045F) }, /* zoo */
 {{000,000,016,001,001,002,004,010,020}, U(10460) }, /* measure */
 {{000,000,016,001,001,002,004,011,026}, U(10461) }, /* judge */
 {{000,000,001,001,002,004,010,020,020}, U(10462) }, /* woe */
 {{000,000,021,012,004,012,021,012,004}, U(10463) }, /* ha-ha */
 {{000,000,017,020,020,020,017,000,000}, U(10464) }, /* loll */
 {{000,000,003,002,004,010,030,000,000}, U(10465) }, /* mime */
 {{000,000,004,004,004,004,004,000,000}, U(10466) }, /* if */
 {{000,000,004,004,004,004,003,000,000}, U(10467) }, /* egg */
 {{000,000,004,004,004,004,030,000,000}, U(10468) }, /* ash */
 {{000,000,003,004,004,004,004,000,000}, U(10469) }, /* ado */
 {{000,000,030,004,004,004,004,000,000}, U(1046A) }, /* on */
 {{000,000,021,021,012,012,004,000,000}, U(1046B) }, /* wool */
 {{000,000,004,010,034,004,002,000,000}, U(1046C) }, /* out */
 {{000,000,006,010,007,002,034,000,000}, U(1046D) }, /* ah */
 {{000,000,036,001,001,001,036,000,000}, U(1046E) }, /* roar */
 {{000,000,030,010,004,002,003,000,000}, U(1046F) }, /* nun */
 {{000,000,020,023,025,031,001,000,000}, U(10470) }, /* eat */
 {{000,000,037,020,010,004,003,000,000}, U(10471) }, /* age */
 {{000,000,037,001,002,004,030,000,000}, U(10472) }, /* ice */
 {{000,000,037,001,016,020,020,000,000}, U(10473) }, /* up */
 {{000,000,016,021,021,021,016,000,000}, U(10474) }, /* oak */
 {{000,000,004,012,012,021,021,000,000}, U(10475) }, /* ooze */
 {{000,000,030,007,002,004,010,000,000}, U(10476) }, /* oil */
 {{000,000,034,003,001,002,001,000,000}, U(10477) }, /* awe */
 {{000,000,016,022,021,011,032,000,000}, U(10478) }, /* are */
 {{000,000,013,022,021,011,016,000,000}, U(10479) }, /* or */
 {{000,000,016,021,011,021,022,000,000}, U(1047A) }, /* air */
 {{000,000,011,021,022,021,016,000,000}, U(1047B) }, /* err */
 {{000,000,006,011,021,021,026,000,000}, U(1047C) }, /* array */
 {{000,000,026,031,021,021,026,000,000}, U(1047D) }, /* ear */
 {{000,000,027,030,020,020,020,000,000}, U(1047E) }, /* ian */
 {{000,000,020,020,022,025,015,000,000}, U(1047F) }, /* yew */

 /* Musical symbols */
 {{004,004,004,004,004,004,004,004,004}, U(1D100) }, /* bar line */
 {{012,012,012,012,012,012,012,012,012}, U(1D101) }, /* double bar line */
 {{013,013,013,013,013,013,013,013,013}, U(1D102) }, /* final bar line */
 {{032,032,032,032,032,032,032,032,032}, U(1D103) }, /* reversed ditto */
 {{004,004,000,004,004,004,000,004,004}, U(1D104) }, /* dashed bar line */
 {{000,000,004,004,004,004,004,000,000}, U(1D105) }, /* short bar line */
 {{024,024,024,025,024,025,024,024,024}, U(1D106) }, /* start repeat */
 {{005,005,005,025,005,025,005,005,005}, U(1D107) }, /* end repeat */
 {{000,000,000,004,000,004,000,000,000}, U(1D108) }, /* repeat dots */
 {{010,024,021,012,004,012,021,005,002}, U(1D10B) }, /* segno */
 {{016,021,025,000,000,000,000,000,000}, U(1D110) }, /* fermata */
 {{000,000,000,000,000,000,025,021,016}, U(1D111) }, /* fermata below */
 {{000,000,000,000,037,000,000,000,000}, U(1D116) }, /* 1-line stave */
 {{000,000,000,037,000,037,000,000,000}, U(1D117) }, /* 2-line stave */
 {{000,000,037,000,037,000,037,000,000}, U(1D118) }, /* 3-line stave */
 {{000,037,000,037,000,037,000,037,000}, U(1D119) }, /* 4-line stave */
 {{037,000,037,000,037,000,037,000,037}, U(1D11A) }, /* 5-line stave */
 {{004,012,012,012,014,026,025,016,004}, U(1D11E) }, /* G clef */
 {{025,025,025,026,024,026,025,025,025}, U(1D121) }, /* C clef */
 {{010,025,004,005,010,020,000,000,000}, U(1D122) }, /* F clef */
 {{000,033,033,004,033,033,000,000,000}, U(1D12A) }, /* double sharp */
 {{004,024,027,035,025,026,030,000,000}, U(1D12B) }, /* double flat */
 {{000,000,037,016,037,000,000,000,000}, U(1D13A) }, /* breve rest */
 {{000,000,037,016,000,000,000,000,000}, U(1D13B) }, /* semibreve rest */
 {{000,000,000,016,037,000,000,000,000}, U(1D13C) }, /* minim rest */

 /* Transport and map symbols */
 {{004,012,012,012,033,033,025,000,000}, U(1F680) }, /* rocket */
 {{000,034,010,011,027,010,000,000,000}, U(1F681) }, /* helicopter */
 {{000,000,013,036,037,025,000,000,000}, U(1F682) }, /* steamlocomotive */
 {{000,007,017,027,037,011,000,000,000}, U(1F69A) }, /* deliverytruck */

 /* Symbols for legacy computing */
#define M(x, u) { {x}, U(u), MOS6 }, { {x}, -1, "u" #u ".sep6", SEP6 }
 /* space */     M(0x01, 1FB00), M(0x02, 1FB01), M(0x03, 1FB02),
 M(0x04, 1FB03), M(0x05, 1FB04), M(0x06, 1FB05), M(0x07, 1FB06),
 M(0x08, 1FB07), M(0x09, 1FB08), M(0x0a, 1FB09), M(0x0b, 1FB0A),
 M(0x0c, 1FB0B), M(0x0d, 1FB0C), M(0x0e, 1FB0D), M(0x0f, 1FB0E),
 M(0x10, 1FB0F), M(0x11, 1FB10), M(0x12, 1FB11), M(0x13, 1FB12),
 M(0x14, 1FB13), /* lfblock */   M(0x16, 1FB14), M(0x17, 1FB15),
 M(0x18, 1FB16), M(0x19, 1FB17), M(0x1a, 1FB18), M(0x1b, 1FB19),
 M(0x1c, 1FB1A), M(0x1d, 1FB1B), M(0x1e, 1FB1C), M(0x1f, 1FB1D),
 M(0x20, 1FB1E), M(0x21, 1FB1F), M(0x22, 1FB20), M(0x23, 1FB21),
 M(0x24, 1FB22), M(0x25, 1FB23), M(0x26, 1FB24), M(0x27, 1FB25),
 M(0x28, 1FB26), M(0x29, 1FB27), /* rtblock */   M(0x2b, 1FB28),
 M(0x2c, 1FB29), M(0x2d, 1FB2A), M(0x2e, 1FB2B), M(0x2f, 1FB2C),
 M(0x30, 1FB2D), M(0x31, 1FB2E), M(0x32, 1FB2F), M(0x33, 1FB30),
 M(0x34, 1FB31), M(0x35, 1FB32), M(0x36, 1FB33), M(0x37, 1FB34),
 M(0x38, 1FB35), M(0x39, 1FB36), M(0x3a, 1FB37), M(0x3b, 1FB38),
 M(0x3c, 1FB39), M(0x3d, 1FB3A), M(0x3e, 1FB3B), /* block */
#undef M
 {{030,024,022,021,027,030,000,000,000}, U(1FBB0) }, /* arrowheadptr */
 {{000,012,033,000,033,012,000,000,000}, U(1FBBB) }, /* voided Greek cross */
 {{000,037,001,005,001,037,000,000,000}, U(1FBBC) }, /* right open square dot */
 {{012,025,021,012,021,025,012,000,000}, U(1FBC0) }, /* RISC OS close button */
 {{037,021,035,031,037,033,037,000,000}, U(1FBC4) }, /* invquestion */
 {{004,012,012,004,037,004,004,012,021}, U(1FBC5) }, /* stick figure */
 {{004,012,012,004,025,016,004,012,021}, U(1FBC6) }, /* ... with arms up */
 {{004,012,012,004,015,026,004,012,011}, U(1FBC7) }, /* ... leaning left */
 {{004,012,012,004,026,015,004,012,022}, U(1FBC8) }, /* ... leaning right */
 {{004,012,012,004,037,012,021,037,012}, U(1FBC9) }, /* ... wearing dress */
 {{004,012,021,021,025,033,021,000,000}, U(1FBCA) }, /* white chevron up */
 
 /* Latin small caps */
 ALIAS("a.sc", "uni1D00"), ALIAS("A.c2sc", "a.sc"),
 ALIAS("b.sc", "uni0299"), ALIAS("B.c2sc", "b.sc"),
 ALIAS("c.sc", "uni1D04"), ALIAS("C.c2sc", "c.sc"),
 ALIAS("d.sc", "uni1D05"), ALIAS("D.c2sc", "d.sc"),
 ALIAS("e.sc", "uni1D07"), ALIAS("E.c2sc", "e.sc"),
 ALIAS("f.sc", "uniA730"), ALIAS("F.c2sc", "f.sc"),
 ALIAS("g.sc", "uni0262"), ALIAS("G.c2sc", "g.sc"),
 ALIAS("h.sc", "uni029C"), ALIAS("H.c2sc", "h.sc"),
 ALIAS("i.sc", "uni026A"), ALIAS("I.c2sc", "i.sc"),
 ALIAS("j.sc", "uni1D0A"), ALIAS("J.c2sc", "j.sc"),
 ALIAS("k.sc", "uni1D0B"), ALIAS("K.c2sc", "k.sc"),
 ALIAS("l.sc", "uni029F"), ALIAS("L.c2sc", "l.sc"),
 ALIAS("m.sc", "uni1D0D"), ALIAS("M.c2sc", "m.sc"),
 ALIAS("n.sc", "uni0274"), ALIAS("N.c2sc", "n.sc"),
 ALIAS("o.sc", "uni1D0F"), ALIAS("O.c2sc", "o.sc"),
 ALIAS("p.sc", "uni1D18"), ALIAS("P.c2sc", "p.sc"),
 ALIAS("q.sc", "uniA7AF"), ALIAS("Q.c2sc", "q.sc"),
 ALIAS("r.sc", "uni0280"), ALIAS("R.c2sc", "r.sc"),
 ALIAS("s.sc", "uniA731"), ALIAS("S.c2sc", "s.sc"),
 ALIAS("t.sc", "uni1D1B"), ALIAS("T.c2sc", "t.sc"),
 ALIAS("u.sc", "uni1D1C"), ALIAS("U.c2sc", "u.sc"),
 ALIAS("v.sc", "uni1D20"), ALIAS("V.c2sc", "v.sc"),
 ALIAS("w.sc", "uni1D21"), ALIAS("W.c2sc", "w.sc"),
 {{000,000,021,012,004,012,021,000,000}, -1, "x.sc" }, ALIAS("X.c2sc", "x.sc"),
 ALIAS("y.sc", "uni028F"), ALIAS("Y.c2sc", "y.sc"),
 ALIAS("z.sc", "uni1D22"), ALIAS("Z.c2sc", "z.sc"),
 {{000,000,016,021,022,021,026,000,000}, -1, "germandbls.sc" },
 ALIAS("uni1E9E.c2sc", "germandbls.sc"),
 
 ALIAS("ae.sc", "uni1D01"), ALIAS("AE.c2sc", "ae.sc"),
 ALIAS("eth.sc", "uni1D06"), ALIAS("Eth.c2sc", "eth.sc"),
 ALIAS("oe.sc", "uni0276"), ALIAS("OE.c2sc", "oe.sc"),
 {{000,000,020,036,021,036,020,000,000}, -1, "thorn.sc" },
 ALIAS("Thorn.c2sc", "thorn.sc"),

 /*
  * Backward compatibility aliases.  These are glyphs whose name has
  * changed and where we want to keep the old name working.  All of
  * these were added in Bedstead 002.002.
  */
 ALIAS("uni2126", "Omega"),
 ALIAS("uni2295", "circleplus"),
#define UA(u) ALIAS("uni" #u, "u" #u)
 UA(10450), UA(10451), UA(10452), UA(10453), UA(10454), UA(10455), UA(10456),
 UA(10457), UA(10458), UA(10459), UA(1045A), UA(1045B), UA(1045C), UA(1045D),
 UA(1045E), UA(1045F), UA(10460), UA(10461), UA(10462), UA(10463), UA(10464),
 UA(10465), UA(10466), UA(10467), UA(10468), UA(10469), UA(1046A), UA(1046B),
 UA(1046C), UA(1046D), UA(1046E), UA(1046F), UA(10470), UA(10471), UA(10472),
 UA(10473), UA(10474), UA(10475), UA(10476), UA(10477), UA(10478), UA(10479),
 UA(1047A), UA(1047B), UA(1047C), UA(1047D), UA(1047E), UA(1047F), UA(1F680),
 UA(1F681), UA(1F682), UA(1F69A), UA(1FBB0), UA(1FBBB), UA(1FBBC), UA(1FBC0),
 UA(1FBC4), UA(1FBC5), UA(1FBC6), UA(1FBC7), UA(1FBC8), UA(1FBC9), UA(1FBCA),
#undef UA
#define UA6(u) ALIAS("uni" #u, "u" #u), ALIAS("uni" #u ".sep6", "u" #u ".sep6")
 UA6(1FB00), UA6(1FB01), UA6(1FB02),
 UA6(1FB03), UA6(1FB04), UA6(1FB05), UA6(1FB06),
 UA6(1FB07), UA6(1FB08), UA6(1FB09), UA6(1FB0A),
 UA6(1FB0B), UA6(1FB0C), UA6(1FB0D), UA6(1FB0E),
 UA6(1FB0F), UA6(1FB10), UA6(1FB11), UA6(1FB12),
 UA6(1FB13), UA6(1FB14), UA6(1FB15),
 UA6(1FB16), UA6(1FB17), UA6(1FB18), UA6(1FB19),
 UA6(1FB1A), UA6(1FB1B), UA6(1FB1C), UA6(1FB1D),
 UA6(1FB1E), UA6(1FB1F), UA6(1FB20), UA6(1FB21),
 UA6(1FB22), UA6(1FB23), UA6(1FB24), UA6(1FB25),
 UA6(1FB26), UA6(1FB27), UA6(1FB28),
 UA6(1FB29), UA6(1FB2A), UA6(1FB2B), UA6(1FB2C),
 UA6(1FB2D), UA6(1FB2E), UA6(1FB2F), UA6(1FB30),
 UA6(1FB31), UA6(1FB32), UA6(1FB33), UA6(1FB34),
 UA6(1FB35), UA6(1FB36), UA6(1FB37), UA6(1FB38),
 UA6(1FB39), UA6(1FB3A), UA6(1FB3B),
#undef UA6

 /* and finally */
 {{037,021,021,021,021,021,037,000,000}, -1, ".notdef" },
};

#undef U

#define NGLYPHS (sizeof(glyphs) / sizeof(glyphs[0]))
static int const nglyphs = NGLYPHS;

static struct glyph const *glyphs_by_name[NGLYPHS];

static void dolookups(struct glyph const *);

static int
getpix(char const data[YSIZE], int x, int y, unsigned flags)
{

	if (x < 0 || x >= XSIZE || y < 0 || y >= YSIZE)
		return 0;
	else
		return (data[y] >> (XSIZE - x - 1)) & 1;
}

static bool plottermode = false;

static char * get_fullname()
{
#define FULLNAME_MAX 100
	static char fullname[FULLNAME_MAX];

	sprintf(fullname, "Bedstead%s%s", weight->suffix, width->suffix);
	return fullname;
}

static char * fullname_to_fontname(char const *fullname)
{
#define FONTNAME_MAX 29 /* Adobe-recommended limit */
	static char fontname[FONTNAME_MAX + 1], *op = fontname;
	char const *p = fullname;
	bool gotfamily = false;

	while (*p != '\0') {
		assert(op - fontname <= FONTNAME_MAX);
		if (*p == ' ') {
			if (!gotfamily) {
				*op++ = '-';
				gotfamily = true;
			}
		} else {
			*op++ = *p;
		}
		++p;
	}
	*op++ = '\0';
	return fontname;
}

static int
compare_glyphs_by_name(const void *va, const void *vb)
{
	struct glyph const * const *ap = va, * const *bp = vb;
	struct glyph const *a = *ap, *b = *bp;

	return strcmp(a->name, b->name);
}

static int compare_glyph_to_name(const void *vn, const void *vg)
{
	struct glyph const * const *gp = vg;
	struct glyph const *g = *gp;
	char const *name = vn;

	return strcmp(name, g->name);
}

static struct glyph const *get_glyph_by_name(char const *name)
{
	struct glyph const * const *gp;

	gp = bsearch(name, glyphs_by_name, nglyphs,
		     sizeof(glyphs_by_name[0]), &compare_glyph_to_name);
	assert(gp != NULL);
	return *gp;
}

int
main(int argc, char **argv)
{
	int i;
	int extraglyphs = 0;
	char *endptr;

	if (argc == 2 && strcmp(argv[1], "--complement") == 0) {
		glyph_complement();
		return 0;
	}
	      
	while (argc > 1) {
		for (i = 0; i < nwidths; i++)
			if (strcmp(argv[1], widths[i].option) == 0) {
				width = &widths[i];
				argv++; argc--;
				goto next;
			}
		for (i = 0; i < nweights; i++)
			if (strcmp(argv[1], weights[i].option) == 0) {
				weight = &weights[i];
				argv++; argc--;
				goto next;
			}
		if (strcmp(argv[1], "--plotter") == 0) {
			plottermode = true;
			argv++; argc--;
		} else if (strcmp(argv[1], "--") == 0) {
			argv++; argc--;
			break;
		} else if (argv[1][0] == '-') {
			fprintf(stderr, "unknown option '%s'\n", argv[1]);
			return 1;
		} else break;
		argv++; argc--;
	next:;
	}

        if (argc > 1) {
                char data[YSIZE];
                int i, y;
                unsigned long u;

                for (y = 0; y < YSIZE; y++)
                        data[y] = 0;

                y = 0;
                for (i = 1; i < argc; i++) {
                        if (y >= YSIZE) {
                                fprintf(stderr, "too many arguments\n");
                                return 1;
                        }
                        u = strtoul(argv[i], &endptr, 0);
                        if (u > 077 || !argv[i] || *endptr) {
                                fprintf(stderr, "invalid argument \"%s\"\n",
                                        argv[i]);
                                return 1;
                        }
                        data[y++] = u;
                }
		dochar(data, 0);
                return 0;
        }

	for (i = 0; i < nglyphs; i++)
		if (glyphs[i].unicode == -1)
			extraglyphs++;
	printf("SplineFontDB: 3.0\n");
	printf("FontName: %s\n", fullname_to_fontname(get_fullname()));
	printf("FullName: %s\n", get_fullname());
	printf("FamilyName: Bedstead\n");
	printf("LangName: 1033 \"\" \"Bedstead%s%s\" \"%s\" \"\" \"\" \"\" "
	       "\"\" \"\" \"\" \"\" \"\" \"\" \"\" \"\" \"\" \"\" "
	       "\"%s\" \"%s%s\"\n\n", weight->suffix, width->suffix,
	       "Regular",
	       weight->suffix[0] || width->suffix[0] ? "Bedstead" : "",
	       weight->suffix[0] ? weight->suffix+1 : "",
	       weight->suffix[0] ? width->suffix :
	       width->suffix[0] ? width->suffix+1 : "");
	printf("Weight:%s\n", weight->suffix[0] ? weight->suffix : " Medium");
	printf("OS2_WeightWidthSlopeOnly: 1\n");
	printf("Copyright: Dedicated to the public domain\n");
	printf("Version: 002.002\n");
	printf("ItalicAngle: 0\n");
	printf("UnderlinePosition: %g\n", (double)(-YPIX / 2));
	printf("UnderlineWidth: %g\n", (double)(YPIX));
	printf("OS2StrikeYPos: %d\n", (int)(3 * YPIX));
	printf("OS2StrikeYSize: %d\n", (int)(YPIX));
	printf("Ascent: %g\n", (double)(8 * YPIX));
	printf("Descent: %g\n", (double)(2 * YPIX));
	/* Sub/Superscript are three by five pixels */
	printf("OS2SubXSize: %d\n", (int)(YSIZE * YPIX * 3 / (XSIZE - 1)));
	printf("OS2SupXSize: %d\n", (int)(YSIZE * YPIX * 3 / (XSIZE - 1)));
	printf("OS2SubYSize: %d\n", (int)(YSIZE * YPIX * 5 / (YSIZE - 3)));
	printf("OS2SupYSize: %d\n", (int)(YSIZE * YPIX * 5 / (YSIZE - 3)));
	printf("OS2SubXOff: 0\n");
	printf("OS2SupXOff: 0\n");
	printf("OS2SubYOff: %d\n", (int)(2 * YPIX));
	printf("OS2SupYOff: %d\n", (int)(2 * YPIX));
	printf("FSType: 0\n");
	printf("TTFWeight: %d\n", weight->ttfweight);
	printf("TTFWidth: %d\n", width->ttfwidth);
	dopanose();
	printf("OS2FamilyClass: %d\n", 0x080a);
	printf("LayerCount: 2\n");
	printf("Layer: 0 0 \"Back\" 1\n");
	printf("Layer: 1 0 \"Fore\" 0\n");
	printf("Encoding: UnicodeFull\n");
	printf("NameList: Adobe Glyph List\n");
	printf("DisplaySize: -24\n");
	printf("AntiAlias: 1\n");
	printf("FitToEm: 1\n");
	if (plottermode) {
		printf("StrokedFont: 1\n");
		printf("StrokeWidth: 50\n");
	}
	printf("BeginPrivate: 5\n");
	printf(" StdHW 6 [%4g]\n", (double)YPIX);
	printf(" StdVW 6 [%4g]\n",
	       (double)(XPIX * (100 + weight->weight) / 100));
	printf(" BlueValues 35 [0 0 %4g %4g %4g %4g %4g %4g]\n",
	       (double)(YPIX * 5), (double)(YPIX * 5),
	       (double)(YPIX * 6), (double)(YPIX * 6),
	       (double)(YPIX * 7), (double)(YPIX * 7));
	printf(" OtherBlues 21 [%4g %4g %4g %4g]\n",
	       (double)(YPIX * -2), (double)(YPIX * -2),
	       (double)(YPIX * 1), (double)(YPIX * 1));
	printf(" BlueFuzz 1 0\n"); 
	printf("EndPrivate\n");
	/* Force monochrome at 10 and 20 pixels, and greyscale elsewhere. */
	printf("GaspTable: 5 9 2 10 0 19 3 20 0 65535 3\n");
	printf("Lookup: 3 0 0 \"aalt: all alternates\" {\"aalt\"} "
	    "['aalt' ('DFLT' <'dflt'> 'latn' <'dflt'>)]\n");
	printf("Lookup: 1 0 0 \"ss01: SAA5051 forms\" {\"ss01\" (\"saa5051\")} "
	    "['ss01' ('DFLT' <'dflt'> 'latn' <'dflt'>)]\n");
	printf("OtfFeatName: 'ss01' 1033 \"SAA5051\"\n");
	printf("Lookup: 1 0 0 \"ss02: SAA5052 forms\" {\"ss02\" (\"saa5052\")} "
	    "['ss02' ('DFLT' <'dflt'> 'latn' <'dflt'>)]\n");
	printf("OtfFeatName: 'ss02' 1033 \"SAA5052\"\n");
	printf("Lookup: 1 0 0 \"ss04: SAA5054 forms\" {\"ss04\" (\"saa5054\")} "
	    "['ss04' ('DFLT' <'dflt'> 'latn' <'dflt'>)]\n");
	printf("OtfFeatName: 'ss04' 1033 \"SAA5054\"\n");
	printf("Lookup: 1 0 0 \"ss14: 4-cell separated graphics\" "
	       "{\"ss14\" (\"sep4\")} "
	    "['ss14' ('DFLT' <'dflt'> 'latn' <'dflt'>)]\n");
	printf("OtfFeatName: 'ss14' 1033 \"4-cell separated graphics\"\n");
	printf("Lookup: 1 0 0 \"ss16: 6-cell separated graphics\" "
	       "{\"ss16\" (\"sep6\")} "
	    "['ss16' ('DFLT' <'dflt'> 'latn' <'dflt'>)]\n");
	printf("OtfFeatName: 'ss16' 1033 \"6-cell separated graphics\"\n");
	printf("Lookup: 257 0 0 \"palt: proportional metrics\" {\"palt\"} "
	    "['palt' ('DFLT' <'dflt'> 'latn' <'dflt'>)]\n");
	printf("Lookup: 1 0 0 \"smcp: lower-case to small caps\" "
            "{\"smcp\" (\"sc\")} "
	    "['smcp' ('latn' <'dflt'>)]\n");
	printf("Lookup: 1 0 0 \"c2sc: upper-case to small caps\" "
	    "{\"c2sc\" (\"c2sc\")} "
	    "['c2sc' ('latn' <'dflt'>)]\n");
	printf("BeginChars: %d %d\n", 0x110000 + extraglyphs, nglyphs);
	extraglyphs = 0;
	for (i = 0; i < nglyphs; i++)
		glyphs_by_name[i] = glyphs + i;
	qsort(glyphs_by_name, nglyphs, sizeof(glyphs_by_name[0]),
	      &compare_glyphs_by_name);
	for (i = 0; i < nglyphs; i++) {
		printf("\nStartChar: %s\n", glyphs[i].name);
		printf("Encoding: %d %d %d\n",
		    glyphs[i].unicode != -1 ? glyphs[i].unicode :
		    0x110000 + extraglyphs++, glyphs[i].unicode, i);
		printf("Width: %g\n", (double)(XSIZE * XPIX));
		if (glyphs[i].flags & (MOS6|MOS4))
			printf("Flags: W\n");
		else
			printf("Flags: HW\n");
		printf("LayerCount: 2\n");
		dolookups(&glyphs[i]);
		if (glyphs[i].alias_of != NULL)
			doalias(glyphs[i].alias_of);
		else if	(glyphs[i].flags & MOS6)
			domosaic(glyphs[i].data[0],
				 (glyphs[i].flags & SEP) != 0);
		else if (glyphs[i].flags & MOS4)
			domosaic4(glyphs[i].data[0],
				  (glyphs[i].flags & SEP) != 0);
		else {
			if (plottermode)
				dochar_plotter(glyphs[i].data, glyphs[i].flags);
			else
				dochar(glyphs[i].data, glyphs[i].flags);
		}
		printf("EndChar\n");
	}
	printf("EndChars\n");
	printf("EndSplineFont\n");
	return 0;
}

static void
dopanose()
{
	/*
	 * PANOSE is a complex font categorisation scheme.  I suspect
	 * no-one uses it, but it's a mandatory part of the 'OS/2'
	 * table, so it would be nice to get it right.  This procedure
	 * calculates the PANOSE code for a Bedstead variant based on
	 * its design parameters.
	 *
	 * See <http://panose.com> for the full details.
	 */
	int panose[10];
	int stdvw;
	/* Common digits */
	/* WeightRat == 700/StdVW */
	stdvw = XPIX * (100 + weight->weight) / 100;
	if (stdvw <= 20) /* WeightRat >= 35 */
		panose[2] = 2; /* Very Light */
	else if (stdvw < 39) /* WeightRat > 17.9 */
		panose[2] = 3; /* Light */
	else if (stdvw <= 70) /* WeightRat >= 10 */
		panose[2] = 4; /* Thin */
	else if (stdvw <= 94) /* WeightRat > 7.44 */
		panose[2] = 5; /* Book */
	else if (stdvw < 128) /* WeightRat > 5.5 */
		panose[2] = 6; /* Medium */
	else if (stdvw < 156) /* WeightRat > 4.5 */
		panose[2] = 7; /* Demi */
	else if (stdvw <= 200) /* WeightRat >= 3.5 */
		panose[2] = 8; /* Bold */
	else if (stdvw <= 280) /* WeightRat >= 2.5 */
		panose[2] = 9; /* Heavy */
	else if (stdvw <= 350) /* WeightRat >= 2.0 */
		panose[2] = 10; /* Black */
	else
		panose[2] = 11; /* Extra Black */
	/*
	 * To make life simpler, ignore diagonals when calculating
	 * ConRat.
	 */
	/* ConRat == min(StdVW / 100, 100 / StdVW) */
	if (stdvw > 80 && stdvw < 125)
		panose[4] = 2; /* None */
	else if (stdvw > 65 && stdvw < 154)
		panose[4] = 3; /* Very Low */
	else if (stdvw > 48 && stdvw < 209)
		panose[4] = 4; /* Low */
	else if (stdvw > 30 && stdvw < 334)
		panose[4] = 5; /* Medium Low */
	else if (stdvw > 20 && stdvw < 500)
		panose[4] = 6; /* Medium */
	else if (stdvw > 15 && stdvw < 667)
		panose[4] = 7; /* Medium High */
	else if (stdvw > 8 && stdvw < 1250)
		panose[4] = 8; /* High */
	else
		panose[4] = 9; /* Very High */
	/*
	 * First digit is overall style.  Bedstead is a text font, but
	 * PANOSE says that fonts with horizontal stress and extreme
	 * aspect ratios should be classified as decorative.
	 */
	if (stdvw < 100 || /* ConRat > 1.00 */
	    XPIX > 164) { /* ORat < 0.85 */
		panose[0] = 4; /* Latin Decorative */
		panose[1] = 2; /* Derivative */
		/* ORat == 140/XPIX */
		if (XPIX <= 53)
			panose[3] = 2; /* Super Condensed */
		else if (XPIX <= 66)
			panose[3] = 3; /* Very Condensed */
		else if (XPIX <= 110)
			panose[3] = 4; /* Condensed */
		else if (XPIX <= 152)
			panose[3] = 5; /* Normal */
		else if (XPIX <= 155)
			panose[3] = 6; /* Extended */
		else if (XPIX <= 164)
			panose[3] = 7; /* Very Extended */
		else
			panose[3] = 8; /* Super Extended */
		panose[5] = 11; /* Normal Sans Serif */
		panose[6] = 2; /* Standard Solid Fill */
		panose[7] = 2; /* No Lining */
		panose[8] = 3; /* Square */
		panose[9] = 2; /* Extended Collection */
	} else {
		panose[0] = 2; /* Latin Text */
		panose[1] = 11; /* Normal Sans Serif */
		/* J and M are the same width. */
		panose[3] = 9; /* Monospaced */
		if (panose[4] == 2)
			panose[5] = 2; /* No Variation */
		else if (stdvw > 100)
			panose[5] = 5; /* Gradual/Vertical */
		else
			panose[5] = 6; /* Gradual/Hoizontal */
		/* Unusually shaped 'A' means no fit here. */
		panose[6] = 1; /* No Fit */
		/* OutCurv == 0.791 */
		/* FIXME: Only correct for weight == 0 */
		panose[7] = 5; /* Normal/Boxed */
		panose[8] = 3; /* Standard/Pointed */
		panose[9] = 7; /* Ducking/Large */
	}
	printf("Panose: %d %d %d %d %d %d %d %d %d %d\n",
	       panose[0], panose[1], panose[2], panose[3], panose[4], 
	       panose[5], panose[6], panose[7], panose[8], panose[9]);
}

static void
dopalt(struct glyph const *g)
{
	int i;
	unsigned char cols = 0;
	int dx = 0, dh = 0;

	while (g->alias_of != NULL)
		g = get_glyph_by_name(g->alias_of);
	if (g->flags & (MOS6|MOS4)) return;
	/*
	 * For proportional layout, we'd like a left side-bearing of
	 * one pixel, and a right side-bearing of zero.  Space
	 * characters get an advance width of three pixels.
	 */
	for (i = 0; i < YSIZE; i++)
		cols |= g->data[i] & ((1 << XSIZE) - 1);
	if (cols == 0)
		dh = 3 - XSIZE;
	else {
		while (!(cols & 1 << (XSIZE - 2))) {
			cols <<= 1;
			dx--;
		}
		while (!(cols & 1)) {
			cols >>= 1;
			dh--;
		}
	}
	if (dx || dh)
		printf("Position2: \"palt\" dx=%d dy=0 dh=%d dv=0\n",
		       (int)(dx * XPIX), (int)(dh * XPIX));
}


static void
dolookups(struct glyph const *g)
{
	char prefix[32];
	struct glyph const **found, **gp;
	size_t plen;
	bool any_alt = false;

	plen = sprintf(prefix, "%s.", g->name);
	assert(plen < 32);

	/* Look for related glyphs */
	found = bsearch(&g, glyphs_by_name, nglyphs, sizeof(glyphs_by_name[0]),
		&compare_glyphs_by_name);
	assert(found != NULL);
	for (gp = found + 1; gp < glyphs_by_name + nglyphs; gp++) {
		if (strncmp(prefix, (*gp)->name, plen) != 0) break;
		any_alt = true;
		if (strcmp((*gp)->name + plen, "saa5051") == 0)
			printf("Substitution2: \"ss01\" %s\n", (*gp)->name);
		if (strcmp((*gp)->name + plen, "saa5052") == 0)
			printf("Substitution2: \"ss02\" %s\n", (*gp)->name);
		if (strcmp((*gp)->name + plen, "saa5054") == 0)
			printf("Substitution2: \"ss04\" %s\n", (*gp)->name);
		if (strcmp((*gp)->name + plen, "sep4") == 0)
			printf("Substitution2: \"ss14\" %s\n", (*gp)->name);
		if (strcmp((*gp)->name + plen, "sep6") == 0)
			printf("Substitution2: \"ss16\" %s\n", (*gp)->name);
		if (strcmp((*gp)->name + plen, "sc") == 0)
			printf("Substitution2: \"smcp\" %s\n", (*gp)->name);
		if (strcmp((*gp)->name + plen, "c2sc") == 0)
			printf("Substitution2: \"c2sc\" %s\n", (*gp)->name);
	}
	if (any_alt) {
		printf("AlternateSubs2: \"aalt\"");
		for (gp = found + 1; gp < glyphs_by_name + nglyphs; gp++) {
			if (strncmp(prefix, (*gp)->name, plen) != 0) break;
			printf(" %s", (*gp)->name);
		}
		printf("\n");
	}
	dopalt(g);
}

typedef struct vec {
	int x, y;
} vec;

typedef struct point {
	struct point *next, *prev;
	struct vec v;
} point;

#define MAXPOINTS (XSIZE * YSIZE * 20)

static point points[MAXPOINTS];

static int nextpoint;

static void
clearpath()
{

	nextpoint = 0;
}

static void
moveto(unsigned x, unsigned y)
{
	struct point *p = &points[nextpoint++];

	p->v.x = x; p->v.y = y;
	p->next = p->prev = NULL;
}

static void
lineto(unsigned x, unsigned y)
{
	struct point *p = &points[nextpoint++];

	p->v.x = x; p->v.y = y;
	p->next = NULL;
	p->prev = p - 1;
	p->prev->next = p;
}

static void
closepath()
{
	struct point *p = &points[nextpoint - 1];

	while (p->prev) p--;
	p->prev = points + nextpoint - 1;
	points[nextpoint - 1].next = p;
}

static void
killpoint(point *p)
{

	p->prev->next = p->next;
	p->next->prev = p->prev;
	p->next = p->prev = NULL;
}

static vec const zero = { 0, 0 };

static int
vec_eqp(vec v1, vec v2)
{
	return v1.x == v2.x && v1.y == v2.y;
}

static vec
vec_sub(vec v1, vec v2)
{
	vec ret;
	ret.x = v1.x - v2.x; ret.y = v1.y - v2.y;
	return ret;
}

static int
gcd(int a, int b)
{
	int t;
	while (b != 0) {
		t = b;
		b = a % b;
		a = t;
	}
	return a;
}

/* Return the shortest representable vector parallel to the argument. */
static vec
vec_bearing(vec v)
{
	vec ret;
	int d = gcd(abs(v.x), abs(v.y));
	if (d != 0) {
		ret.x = v.x / d;
		ret.y = v.y / d;
	} else {
		ret.x = 0;
		ret.y = 0;
	}
	return ret;
}

/* If p is identical to its successor, remove p. */
static void
fix_identical(point *p)
{
	if (!p->next) return;
	if (vec_eqp(p->next->v, p->v))
		killpoint(p);
}

/* Are a, b, and c distinct collinear points in that order? */
static int
vec_inline3(vec a, vec b, vec c)
{
	return
	    vec_eqp(vec_bearing(vec_sub(b, a)), vec_bearing(vec_sub(c, b))) &&
	    !vec_eqp(vec_bearing(vec_sub(b, a)), zero);
}

/* Are a, b, c, and d distinct collinear points in that order? */
static int
vec_inline4(vec a, vec b, vec c, vec d)
{
	return vec_inline3(a, b, c) && vec_inline3(b, c, d);
}

/* If p is on the line between its predecessor and successor, remove p. */
static void
fix_collinear(point *p)
{
	if (!p->next) return;
	if (vec_inline3(p->prev->v, p->v, p->next->v))
		killpoint(p);
}

/* If p is the only point on its path, remove p. */
static void
fix_isolated(point *p)
{
	if (p->next == p)
		killpoint(p);
}

static int done_anything;

static void
fix_edges(point *a0, point *b0)
{
	point *a1 = a0->next, *b1 = b0->next;

	assert(a1->prev == a0); assert(b1->prev == b0);
	assert(a0 != a1); assert(a0 != b0);
	assert(a1 != b1); assert(b0 != b1);
	if (vec_eqp(vec_bearing(vec_sub(a0->v, a1->v)),
		    vec_bearing(vec_sub(b1->v, b0->v))) &&
	    (vec_inline4(a0->v, b1->v, a1->v, b0->v) ||
	     vec_inline4(a0->v, b1->v, b0->v, a1->v) ||
	     vec_inline4(b1->v, a0->v, b0->v, a1->v) ||
	     vec_inline4(b1->v, a0->v, a1->v, b0->v) ||
	     vec_eqp(a0->v, b1->v) || vec_eqp(a1->v, b0->v))) {
		a0->next = b1; b1->prev = a0;
		b0->next = a1; a1->prev = b0;
		fix_isolated(a0);
		fix_identical(a0);
		fix_collinear(b1);
		fix_isolated(b0);
		fix_identical(b0);
		fix_collinear(a1);
		done_anything = 1;
	}
}

static void
clean_path()
{
	int i, j;

	do {
		done_anything = 0;
		for (i = 0; i < nextpoint; i++)
			for (j = i+1; points[i].next && j < nextpoint; j++)
				if (points[j].next)
					fix_edges(&points[i], &points[j]);
	} while (done_anything);
}

static void
emit_contour(point *p0)
{
	point *p = p0, *p1;

	do {
		printf(" %g %g %s 1\n",
		       (double)p->v.x / XSCALE,
		       (double)p->v.y / YSCALE - 3*YPIX,
		       p == p0 && p->next ? "m" : "l");
		p1 = p->next;
		p->prev = p->next = NULL;
		p = p1;
	} while (p);
}

static void 
emit_path()
{
	int i, pass;
	point *p;
	bool started = false;

	/*
	 * On the first pass, emit open contours (if there are any).
	 * On the second pass, emit all remaining contours.
	 */
	for (pass = 0; pass <= 1; pass++) {
		for (i = 0; i < nextpoint; i++) {
			p = &points[i];
			if (p->next && (!p->prev || pass == 1)) {
				if (!started) printf("Fore\nSplineSet\n");
				started = true;
				emit_contour(p);
			}
		}
	}
	if (started) printf("EndSplineSet\n");
}

/*
 * To vary the weight of Bedstead, we just vary the thickness of
 * vertical strokes.  More precisely, we pretend that the weight
 * variation is achieved by moving the rising edges of the output
 * waveform of the SAA5050.  This is implemented by moving all left
 * edges left and right.  The code below only handles cases where we
 * don't run out of slack in horizontal lines, which limits weight to
 * the range (-0.5 * XPIX) < weight < (0.25 * XPIX).
 */
static void
adjust_weight()
{
	int i;
	point *p;

	for (i = 0; i < nextpoint; i++) {
		p = &points[i];
		if (p->next == NULL) continue;
		assert(p->prev != NULL);
		/* Move left-edge points horizontally */
		if (p->prev->v.y <= p->v.y && p->v.y <= p->next->v.y)
			p->v.x -= weight->weight;
		/* Move top inner corner points along NE/SW diagonal */
		if (p->prev->v.y < p->v.y && p->v.y > p->next->v.y &&
		    p->prev->v.x > p->v.x && p->v.x > p->next->v.x) {
			p->v.x -= weight->weight/2;
			p->v.y -= weight->weight/2;
		}
		/* Move bottom inner corner points along NW/SE diagonal */
		if (p->prev->v.y > p->v.y && p->v.y < p->next->v.y &&
		    p->prev->v.x < p->v.x && p->v.x < p->next->v.x) {
			p->v.x -= weight->weight/2;
			p->v.y += weight->weight/2;
		}
	}
}

static void
blackpixel(int x, int y, int bl, int br, int tr, int tl)
{
	x *= XPIX_S; y *= YPIX_S;

	if (bl)	moveto(x, y);
	else { moveto(x+XQTR_S, y); lineto(x, y+YQTR_S); }
	if (tl) lineto(x, y+YPIX_S);
	else { lineto(x, y+YPIX_S-YQTR_S); lineto(x+XQTR_S, y+YPIX_S); }
	if (tr) lineto(x+XPIX_S, y+YPIX_S);
	else { lineto(x+XPIX_S-XQTR_S, y+YPIX_S);
	       lineto(x+XPIX_S, y+YPIX_S-YQTR_S); }
	if (br) lineto(x+XPIX_S, y);
	else { lineto(x+XPIX_S, y+YQTR_S); lineto(x+XPIX_S-XQTR_S, y); }
	closepath();
}

static void
whitepixel(int x, int y, int bl, int br, int tr, int tl)
{
	x *= XPIX_S; y *= YPIX_S;

	if (bl) {
		moveto(x, y); lineto(x, y+YPIX_S-YQTR_S);
		if (br) { lineto(x+XPIX_S/2, y+YPIX_S/2-YQTR_S);
			  lineto(x+XQTR_S, y); }
		else lineto(x+XPIX_S-XQTR_S, y);
		closepath();
	}
	if (tl) {
		moveto(x, y+YPIX_S); lineto(x+XPIX_S-XQTR_S, y+YPIX_S);
		if (bl) { lineto(x+XPIX_S/2-XQTR_S, y+YPIX_S/2);
			lineto(x, y+YPIX_S-YQTR_S); }
		else lineto(x, y+YQTR_S);
		closepath();
	}
	if (tr) {
		moveto(x+XPIX_S, y+YPIX_S); lineto(x+XPIX_S, y+YQTR_S);
		if (tl) { lineto(x+XPIX_S/2, y+YPIX_S/2+YQTR_S);
			lineto(x+XPIX_S-XQTR_S, y+YPIX_S); }
		else lineto(x+XQTR_S, y+YPIX_S);
		closepath();
	}
	if (br) {
		moveto(x+XPIX_S, y); lineto(x+XQTR_S, y);
		if (tr) { lineto(x+XPIX_S/2+XQTR_S, y+YPIX_S/2);
			lineto(x+XPIX_S, y+YQTR_S); }
		else lineto(x+XPIX_S, y+YPIX_S-YQTR_S);
		closepath();
	}
}

static void
dochar(char const data[YSIZE], unsigned flags)
{
	int x, y;

#define GETPIX(x,y) (getpix(data, (x), (y), flags))
#define L GETPIX(x-1, y)
#define R GETPIX(x+1, y)
#define U GETPIX(x, y-1)
#define D GETPIX(x, y+1)
#define UL GETPIX(x-1, y-1)
#define UR GETPIX(x+1, y-1)
#define DL GETPIX(x-1, y+1)
#define DR GETPIX(x+1, y+1)

	clearpath();
	for (x = 0; x < XSIZE; x++) {
		for (y = 0; y < YSIZE; y++) {
			if (GETPIX(x, y)) {
				bool tl, tr, bl, br;

				/* Assume filled in */
				tl = tr = bl = br = true;
				/* Check for diagonals */
				if ((UL && !U && !L) || (DR && !D && !R))
					tr = bl = false;
				if ((UR && !U && !R) || (DL && !D && !L))
					tl = br = false;
				/* Avoid odd gaps */
				if (L || UL || U) tl = true;
				if (R || UR || U) tr = true;
				if (L || DL || D) bl = true;
				if (R || DR || D) br = true;
				blackpixel(x, YSIZE - y - 1, bl, br, tr, tl);
			} else {
				bool tl, tr, bl, br;

				/* Assume clear */
				tl = tr = bl = br = false;
				/* white pixel -- just diagonals */
				if (L && U && !UL) tl = true;
				if (R && U && !UR) tr = true;
				if (L && D && !DL) bl = true;
				if (R && D && !DR) br = true;
				whitepixel(x, YSIZE - y - 1, bl, br, tr, tl);
			}
		}
	}
	clean_path();
	adjust_weight();
	emit_path();
}

static void
reverse_contour(point *a)
{
	point *tmp;
	
	while (a->prev != NULL)
		a = a->prev;
	while (a != NULL) {
		tmp = a->next;
		a->next = a->prev;
		a->prev = tmp;
		a = tmp;
	}
}		

/* Join together two points each at the end of a contour */
static void
join_ends(point *a, point *b)
{
	point *tmp;

	if (a->prev == NULL && b->next == NULL) {
		tmp = a; a = b; b = tmp;
	}
	if (a->prev == NULL)
		reverse_contour(a);
	if (b->next == NULL)
		reverse_contour(b);
	assert(a->next == NULL);
	assert(a->prev != NULL);
	assert(b->prev == NULL);
	assert(b->next != NULL);
	assert(vec_eqp(a->v, b->v));
	a->next = b;
	b->prev = a;
	fix_identical(a); /* Will delete a */
	fix_collinear(b); /* Might delete b */
}

static bool
point_endp(point *p)
{
	return p->next == NULL || p->prev == NULL;
}

static bool
point_deadp(point *p)
{
	return p->next == NULL && p->prev == NULL;
}

static void
clean_skeleton()
{
	int i, j;

	/* Pass 1: join collinear connected segments */
	for (i = 0; i < nextpoint; i++) {
		if (point_deadp(&points[i]))
			continue;
		for (j = 0; j < nextpoint; j++) {
			if (point_deadp(&points[j]))
				continue;
			if (vec_eqp(points[i].v, points[j].v) &&
			    points[i].next == NULL && points[j].prev == NULL &&
			    vec_inline3(points[i].prev->v, points[i].v,
					points[j].next->v))
				join_ends(&points[i], &points[j]);
			if (point_deadp(&points[i]))
				break;
		}
	}
	/* Pass 2: join any connected segments */
	for (i = 0; i < nextpoint; i++) {
		if (point_deadp(&points[i]))
			continue;
		for (j = i+1; j < nextpoint; j++) {
			if (point_deadp(&points[j]))
				continue;
			if (vec_eqp(points[i].v, points[j].v) &&
			    point_endp(&points[i]) && point_endp(&points[j]))
				join_ends(&points[i], &points[j]);
			if (point_deadp(&points[i]))
				break;
		}
	}
}

static void
dochar_plotter(char const data[YSIZE], unsigned flags)
{
	int x, y;

#define CONNECT(dx, dy) do { \
		moveto(x * XPIX_S, (YSIZE-y-1) * YPIX_S); \
		lineto((x+dx) * XPIX_S, (YSIZE-y-1+dy) * YPIX_S); \
	} while (0)
	
	clearpath();
	for (x = 0; x < XSIZE; x++) {
		for (y = 0; y < YSIZE; y++) {
			if (GETPIX(x, y)) {
				if (R) CONNECT(1, 0);
				if (D) CONNECT(0, -1);
				if (DR && !D && !R) CONNECT(1, -1);
				if (DL && !D && !L) CONNECT(-1, -1);
				if (!U && !D && !L && !R &&
				    !UL && !UR && !DL && !DR)	{
					/* draw a dot */
					CONNECT(0, 0);
				}
			}
		}
	}
	clean_skeleton();
	emit_path();
}

static void
tile(int x0, int y0, int x1, int y1)
{
	x0 *= XPIX_S; y0 *= YPIX_S;
	x1 *= XPIX_S; y1 *= YPIX_S;
	moveto(x0, y0); lineto(x0, y1); lineto(x1, y1); lineto(x1, y0);
	closepath();
}
	
static void
domosaic(unsigned code, bool sep)
{

	clearpath();
	if (code & 1)  tile(0 + sep, 8 + sep, 3, 11);
	if (code & 2)  tile(3 + sep, 8 + sep, 6, 11);
	if (code & 4)  tile(0 + sep, 4 + sep, 3, 8);
	if (code & 8)  tile(3 + sep, 4 + sep, 6, 8);
	if (code & 16) tile(0 + sep, 1 + sep, 3, 4);
	if (code & 32) tile(3 + sep, 1 + sep, 6, 4);
	clean_path();
	emit_path();
}

static void
domosaic4(unsigned code, bool sep)
{

	clearpath();
	if (code & 1) tile(0 + sep, 6 + sep, 3, 11);
	if (code & 2) tile(3 + sep, 6 + sep, 6, 11);
	if (code & 4) tile(0 + sep, 1 + sep, 3, 6);
	if (code & 8) tile(3 + sep, 1 + sep, 6, 6);
	clean_path();
	emit_path();
}

static void
doalias(char const *alias_of)
{
	struct glyph const *g;

	g = get_glyph_by_name(alias_of);
	printf("Refer: %td %d N 1 0 0 1 0 0 1\n", g - glyphs, g->unicode);
}
	
static int
byunicode(const void *va, const void *vb)
{
	struct glyph const *a = *(struct glyph const **)va,
		*b = *(struct glyph const **)vb;

	/* Cast to unsigned so -1 sorts last. */
	if ((unsigned)a->unicode < (unsigned)b->unicode) return -1;
	if ((unsigned)a->unicode > (unsigned)b->unicode) return +1;
	return strcmp(a->name, b->name);
}
	
static void
glyph_complement()
{
	int const nrow = 16, ncol=12;
	int i, unicol = 32/nrow, col = -1, row = 0;
	int const nglyphs = sizeof(glyphs) / sizeof(glyphs[0]);
	int npages = 0;
	bool newcol = false;
	struct glyph const *sorted[nglyphs], *g;

	for (i = 0; i < nglyphs; i++)
		sorted[i] = &glyphs[i];
	qsort(sorted, nglyphs, sizeof(sorted[0]), &byunicode);
	printf("%%!PS-Adobe\n");
	printf("%%%%Creator: bedstead\n");
	printf("%%%%Title: Bedstead Glyph Complement\n");
	printf("%%%%LanguageLevel: 2\n");
	printf("/xfont /Bedstead findfont 20 scalefont def\n");
	printf("/nfont /Bedstead findfont 10 scalefont def\n");
	printf("/lfont /Bedstead findfont 4 scalefont def\n");
	printf("/str 50 string def\n");
	printf("/centre {\n");
	printf(" dup stringwidth pop 2 div neg 0 rmoveto show\n");
	printf("} def\n");
	printf("/label {\n");
	printf(" lfont setfont centre\n");
	printf("} def\n");
	/* unicode glyphname  exemplify -- */
	printf("/exemplify {\n");
	printf(" xfont setfont dup 14 14 moveto glyphshow\n");
	printf(" 0 0 moveto 0 40 lineto 40 40 lineto 40 0 lineto closepath\n");
	printf("  1 setlinewidth gsave stroke grestore clip\n");
	printf(" 0.1 setlinewidth\n");
	printf("  14 10 moveto 14 30 lineto 26 30 lineto 26 10 lineto\n");
	printf("  closepath stroke\n");
	printf(" 20 36 moveto str cvs label\n");
	printf(" 20 2 moveto label\n");
	printf("} def\n");
	printf("/colnum {\n");
	printf(" nfont setfont 20 42 moveto centre\n");
	printf("} def\n");
	printf("/rownums {\n");
	printf(" nfont setfont 0 1 15 {\n");
	printf("  dup -40 mul 16.5 add -10 exch moveto\n");
	printf("  16 1 string cvrs show\n");
	printf(" } for\n");
	printf("} def\n");
	printf("/setdistillerparams where\n");
	printf("{ pop << /SubsetFonts false >> setdistillerparams } if\n");
	printf("%%%%EndProlog\n");
	for (i = 0; i < nglyphs; i++) {
		g = sorted[i];
		if (g->unicode / nrow != unicol ||
		    (g->unicode == -1 && row == nrow)) {
			if (++col == ncol) {
				printf("grestore showpage\n");
				col = -1;
			}
			unicol = g->unicode / nrow;
			row = 0;
			newcol = true;
		}
		if (col == -1) {
			++npages;
			printf("%%%%Page: %d %d\n", npages, npages);
			printf("gsave 20 700 translate rownums\n");
			col = 0;
			newcol = true;
		}
		if (newcol && g->unicode != -1) {
			printf("gsave %d 0 translate (%03X) colnum grestore\n",
			       col * 40, unicol);
			newcol = false;
		}
		printf("gsave %d %d translate ",
		       (col * 40),
		       -((g->unicode == -1 ? row++ : g->unicode%nrow) * 40));
		if (g->unicode != -1)
			printf("(U+%04X)", g->unicode);
		else
			printf("()");
		printf("/%s ", g->name);
		printf("exemplify grestore\n");
	}
	printf("showpage\n");
	printf("%%%%EOF\n");
}
